<?php
/**
 * WP-CLI functionality
 *
 * @since      1.0.0
 * @package    WVC_Editor
 * @subpackage WVC_Editor/CLI
 */

// Exit if accessed directly
if ( ! defined('ABSPATH')) {
    exit;
}

/**
 * WP-CLI commands for WVC Editor
 */
class WVC_CLI
{

    /**
     * Registers WP-CLI commands
     */
    public static function register_commands()
    {
        if ( ! class_exists('WP_CLI')) {
            return;
        }

        WP_CLI::add_command('wvc', __CLASS__);
    }

    /**
     * Load required manager classes for API operations
     */
    private static function load_manager_classes()
    {
        // Menu Manager
        if ( ! class_exists('WVC_Menu_Manager')) {
            require_once get_template_directory() . '/includes/content-managers/menu-manager.php';
        }

        // Page Manager
        if ( ! class_exists('WVC_Page_Manager')) {
            require_once get_template_directory() . '/includes/content-managers/page-manager.php';
        }

        // Post Manager
        if ( ! class_exists('WVC_Post_Manager')) {
            require_once get_template_directory() . '/includes/content-managers/post-manager.php';
        }

        // Import class
        if ( ! class_exists('WVC_Import')) {
            require_once get_template_directory() . '/includes/import/import.php';
        }
    }

    /**
     * Display plugin version
     *
     * ## EXAMPLES
     *
     *     wp wvc version
     *
     * @when after_wp_load
     */
    public function version()
    {
        WP_CLI::success('WVC version: ' . WVC_EDITOR_VERSION);
    }

    /**
     * Import pages and menus from an S3 zip file
     *
     * ## OPTIONS
     *
     * [--s3_url=<url>]
     * : The S3 signed URL to download the zip file containing pages and menus
     *
     * [--build_id=<id>]
     * : The build id
     *
     * [--format=<format>]
     * : Output format (table, json, csv, yaml)
     * ---
     * default: table
     * options:
     *   - table
     *   - json
     *   - csv
     *   - yaml
     * ---
     *
     * ## EXAMPLES
     *
     *     # Import content from S3 URL
     *     wp wvc import --s3_url=https://s3.amazonaws.com/bucket/path/to/file.zip --build_id=26
     *
     *     # Import and get JSON output
     *     wp wvc import --s3_url=https://s3.amazonaws.com/bucket/path/to/file.zip --build_id=26 --format=json
     *
     * @when after_wp_load
     *
     * @param array $args Command arguments
     * @param array $assoc_args Command associated arguments
     */
    public function import($args, $assoc_args)
    {
        // Ensure we have a URL
        if (empty($assoc_args["s3_url"])) {
            WP_CLI::error('S3 URL is required.');

            return;
        }

        if (empty($assoc_args["build_id"])) {
            WP_CLI::error('Build id is required.');

            return;
        }

        $s3_url   = $assoc_args["s3_url"];
        $build_id = $assoc_args["build_id"];

        // Validate URL
        if ( ! filter_var($s3_url, FILTER_VALIDATE_URL)) {
            WP_CLI::error('Invalid URL format.');

            return;
        }

        // Include the import class if not already included
        if ( ! class_exists('WVC_Import')) {
            require_once get_template_directory() . '/includes/import/import.php';
        }

        WP_CLI::log('Starting import process...');

        try {
            // Initialize the importer
            $importer = new WVC_Import($s3_url, $build_id);

            // Run the import
            $result = $importer->import();
            update_option("wvc_import_result_" . time(), $result);

            if ($result['success']) {
                // Output summary information
                $format = isset($assoc_args['format']) ? $assoc_args['format'] : 'table';

                // Pages summary
                if ( ! empty($result['pages'])) {
                    $pages_summary = [
                        'updated' => $result['pages']['updated'],
                        'failed'  => $result['pages']['failed'],
                        'total'   => count($result['pages']['pages'])
                    ];

                    WP_CLI::log('Pages import summary:');
                    WP_CLI\Utils\format_items($format, [$pages_summary], array_keys($pages_summary));

                    // If table format, also show individual pages
                    if ($format === 'table' && ! empty($result['pages']['pages'])) {
                        WP_CLI::log('Imported pages:');

                        WP_CLI\Utils\format_items($format, $result['pages']['pages'],
                            ['id', 'page_title',]);
                    }
                }

                // Templates summary
                if ( ! empty($result['header_footer'])) {
                    $templates_summary = [
                        'created' => $result['header_footer']['created'],
                        'updated' => $result['header_footer']['updated'],
                        'failed'  => $result['header_footer']['failed'],
                        'total'   => count($result['header_footer']['templates'])
                    ];

                    WP_CLI::log('Templates import summary:');
                    WP_CLI\Utils\format_items($format, [$templates_summary], array_keys($templates_summary));

                    // If table format, also show individual templates
                    if ($format === 'table' && ! empty($result['header_footer']['templates'])) {
                        WP_CLI::log('Imported templates:');

                        WP_CLI\Utils\format_items($format, $result['header_footer']['templates'],
                            ['id', 'template_title', 'template_key']);
                    }
                }

                // Images summary
                if ( ! empty($result['images'])) {
                    $images_summary = [
                        'created' => $result['images']['created'],
                        'updated' => $result['images']['updated'],
                        'failed'  => $result['images']['failed'],
                        'total'   => count($result['images']['images'])
                    ];

                    WP_CLI::log('Images import summary:');
                    WP_CLI\Utils\format_items($format, [$images_summary], array_keys($images_summary));

                    // If table format, also show individual images
                    if ($format === 'table' && ! empty($result['images']['images'])) {
                        WP_CLI::log('Imported images:');

                        WP_CLI\Utils\format_items($format, $result['images']['images'],
                            ['attachment_id', 'status']);
                    }
                }

                // Style Kit summary
                if ( ! empty($result['stylesheet'])) {
                    $style_kit_summary = [
                        'created' => $result['stylesheet']['created'],
                        'updated' => $result['stylesheet']['updated'],
                        'failed'  => $result['stylesheet']['failed'],
                        'message' => $result['stylesheet']['message']
                    ];

                    WP_CLI::log('stylesheet import summary:');
                    WP_CLI\Utils\format_items($format, [$style_kit_summary], array_keys($style_kit_summary));
                }

                // Style Kit summary
                if ( ! empty($result['templates_result'])) {
                    $templates_summary = [
                        'created'   => $result['templates_result']['created'],
                        'updated'   => $result['templates_result']['updated'],
                        'failed'    => $result['templates_result']['failed'],
                        'templates' => $result['templates_result']['templates']
                    ];

                    WP_CLI::log('templates import summary:');
                    WP_CLI\Utils\format_items($format, [$templates_summary], array_keys($templates_summary));
                }

                WP_CLI::success($result['message']);
            } else {
                WP_CLI::error($result['message']);
            }

        } catch (Exception $e) {
            WP_CLI::error('Import failed: ' . $e->getMessage());
            update_option("wvc_import_failed", $e->getMessage());
        }
    }

    /**
     * Execute API operations from REST endpoints via CLI
     *
     * ## OPTIONS
     *
     * <resource>
     * : The resource to operate on (menu, page, post, category, tag, media, settings, delete-restore)
     *
     * <action>
     * : The action to perform (create, get, update, delete, list)
     *
     * [--id=<id>]
     * : Resource ID when performing operations on a specific item
     *
     * [--item_id=<id>]
     * : Resource item ID when performing operations on a specific item
     *
     * [--data=<json>]
     * : JSON string with data for create/update operations
     *
     * [--format=<format>]
     * : Output format (table, json, csv, yaml)
     * ---
     * default: json
     * options:
     *   - table
     *   - json
     *   - csv
     *   - yaml
     * ---
     *
     * ## EXAMPLES
     *
     *     # List all menus
     *     wp wvc api menu list
     *
     *     # Get a specific menu
     *     wp wvc api menu get --id=123
     *
     *     # Create a new menu
     *     wp wvc api menu create --data='{"name":"Main Menu","location":"primary"}'
     *
     *     # Bulk create multiple menus from a JSON file
     *     wp wvc api menu bulk-create --file=/path/to/menus.json
     *
     *     # Bulk create multiple menus from inline JSON
     *     wp wvc api menu bulk-create --data='[{"name":"Main Menu","location":"primary"}, {"name":"Footer Menu","location":"footer"}]'
     *
     *     # Create a menu item
     *     wp wvc api menu create-item --id=123 --data='{"title":"Home","url":"/"}'
     *
     *     # Bulk create multiple menu items
     *     wp wvc api menu bulk-create-item --id=123 --data='[{"title":"Home","url":"/"},{"title":"About","url":"/about"},{"title":"Contact","url":"/contact"}]'
     *
     *     # Bulk create menu items from a JSON file
     *     wp wvc api menu bulk-create-item --id=123 --file=/path/to/menu-items.json
     *
     *     # Delete a menu item
     *     wp wvc api menu delete-item --id=123 --item_id=456
     *
     *     # Bulk delete multiple menus
     *     wp wvc api menu bulk-delete --data='[1, 2, 3]'
     *
     *     # Bulk delete multiple menus from a JSON file
     *     wp wvc api menu bulk-delete --file=/path/to/menu-ids.json
     *
     *     # Bulk delete multiple menus with object format
     *     wp wvc api menu bulk-delete --data='[{"id":1}, {"id":2}, {"id":3}]'
     *
     *     # Bulk delete multiple menu items
     *     wp wvc api menu bulk-delete-item --id=123 --data='[456, 789, 101]'
     *
     *     # Bulk delete multiple menu items from a JSON file
     *     wp wvc api menu bulk-delete-item --id=123 --file=/path/to/item-ids.json
     *
     *     # Bulk delete multiple menu items with object format
     *     wp wvc api menu bulk-delete-item --id=123 --data='[{"item_id":456}, {"item_id":789}, {"item_id":101}]'
     *
     *     # Create a page
     *     wp wvc api page create --data='{"title":"About Us","content":"Page content here"}'
     *
     *     # Bulk create multiple pages
     *     wp wvc api page bulk-create --data='[{"title":"Page 1","content":"Content 1"},{"title":"Page 2","content":"Content 2"}]'
     *
     *     # Update a page
     *     wp wvc api page update --id=123 --data='{"title":"New Title"}'
     *
     *     # Delete a page
     *     wp wvc api page delete --id=123
     *
     *     # Bulk delete multiple pages
     *     wp wvc api page bulk-delete --data='[1, 2, 3]'
     *
     *     # Bulk delete multiple pages with force option
     *     wp wvc api page bulk-delete --data='[{"id":1}, {"id":2}, {"id":3}]' --force=true
     *
     *     # Create a post
     *     wp wvc api post create --data='{"title":"New Post","content":"Post content"}'
     *
     *     # Bulk create multiple posts
     *     wp wvc api post bulk-create --data='[{"title":"Post 1","content":"Content 1"},{"title":"Post 2","content":"Content 2"}]'
     *
     *     # Get post
     *     wp wvc api post get --id=123
     *
     *     # Bulk delete multiple posts
     *     wp wvc api post bulk-delete --data='[1, 2, 3]'
     *
     *     # Bulk delete multiple posts with force option
     *     wp wvc api post bulk-delete --data='[{"id":1}, {"id":2}, {"id":3}]' --force=true
     *
     *     # Create a category
     *     wp wvc api category create --data='{"name":"New Category"}'
     *
     *     # Bulk create multiple categories
     *     wp wvc api category bulk-create --data='[{"name":"Category 1"},{"name":"Category 2","parent":0}]'
     *
     *     # Get category
     *     wp wvc api category get --id=123
     *
     *     # Bulk delete multiple categories
     *     wp wvc api category bulk-delete --data='[1, 2, 3]'
     *
     *     # Bulk delete multiple categories with object format
     *     wp wvc api category bulk-delete --data='[{"id":1}, {"id":2}, {"id":3}]'
     *
     *     # Create a tag
     *     wp wvc api tag create --data='{"name":"New Tag"}'
     *
     *     # Bulk create multiple tags
     *     wp wvc api tag bulk-create --data='[{"name":"Tag 1"},{"name":"Tag 2"}]'
     *
     *     # Bulk delete multiple tags
     *     wp wvc api tag bulk-delete --data='[1, 2, 3]'
     *
     *     # Bulk delete multiple tags with object format
     *     wp wvc api tag bulk-delete --data='[{"id":1}, {"id":2}, {"id":3}]'
     *
     *     # Upload media
     *     wp wvc api media upload --data='{"file":"path/to/file.jpg"}'
     *
     *     # Get site settings
     *     wp wvc api settings get
     *
     *     # Update site settings
     *     wp wvc api settings update --data='{"blogname":"New Site Title"}'
     *
     *     # Delete a post (to trash)
     *     wp wvc api delete-restore delete --data='{"type":"post","id":123}'
     *
     *     # Restore a post from trash
     *     wp wvc api delete-restore restore --data='{"type":"post","id":123}'
     *
     *     # Bulk delete multiple items
     *     wp wvc api delete-restore bulk-delete --data='[{"type":"post","id":123},{"type":"post","id":456}]'
     *
     *     # Bulk restore multiple items
     *     wp wvc api delete-restore bulk-restore --data='[{"type":"post","id":123},{"type":"post","id":456}]'
     *
     *     # Reset the site
     *     wp wvc api site reset --confirm --force
     *
     * @when after_wp_load
     *
     * @param array $args Command arguments
     * @param array $assoc_args Command associated arguments
     */
    public function api($args, $assoc_args)
    {
        // Make sure we have required arguments
        if (count($args) < 2) {
            WP_CLI::error('Missing required arguments. Usage: wp wvc api <resource> <action>');

            return;
        }

        $resource = $args[0];
        $action   = $args[1];

        // Load necessary classes
        self::load_manager_classes();

        // Parse JSON data if provided
        $data = [];
        if ( ! empty($assoc_args['data'])) {
            $data = json_decode($assoc_args['data'], true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                WP_CLI::error('Invalid JSON data: ' . json_last_error_msg());

                return;
            }
        }

        // Get format for output
        $format = isset($assoc_args['format']) ? $assoc_args['format'] : 'json';

        // Execute appropriate action based on resource and action
        $result = null;
        try {
            switch ($resource) {
                case 'menu':
                    $result = $this->handle_menu_api($action, $assoc_args, $data);
                    break;

                case 'page':
                    $result = $this->handle_page_api($action, $assoc_args, $data);
                    break;

                case 'post':
                    $result = $this->handle_post_api($action, $assoc_args, $data);
                    break;

                case 'category':
                    $result = $this->handle_category_api($action, $assoc_args, $data);
                    break;

                case 'tag':
                    $result = $this->handle_tag_api($action, $assoc_args, $data);
                    break;

                case 'media':
                    $result = $this->handle_media_api($action, $assoc_args, $data);
                    break;

                case 'settings':
                    $result = $this->handle_settings_api($action, $assoc_args, $data);
                    break;

                case 'delete-restore':
                    $result = $this->handle_delete_restore_api($action, $assoc_args, $data);
                    break;

                case 'site':
                    $result = $this->handle_site_api($action, $assoc_args, $data);
                    break;

                default:
                    WP_CLI::error("Unknown resource: $resource. Available resources: menu, page, post, category, tag, media, settings, delete-restore, import");

                    return;
            }

            // Output the result
            if ($result instanceof WP_Error) {
                WP_CLI::error($result->get_error_message());
            } else {
                if ($format === 'table' && is_array($result)) {
                    if (isset($result[0]) && is_array($result[0])) {
                        // Multiple items - display as table with all keys from first item
                        $keys = array_keys($result[0]);
                        WP_CLI\Utils\format_items($format, $result, $keys);
                    } else {
                        // Single item - flatten for table display
                        $flat_result = $this->flatten_array($result);
                        WP_CLI\Utils\format_items($format, [$flat_result], array_keys($flat_result));
                    }
                } else {
                    // For non-table formats, use WP_CLI::log with the formatted result
                    WP_CLI::log(json_encode($result, JSON_PRETTY_PRINT));
                }
            }

        } catch (Exception $e) {
            WP_CLI::error($e->getMessage());
        }
    }
    /**
     * Handle site API operations
     *
     * @param string $action Action to perform
     * @param array $args Command arguments
     * @param array $data Data for create/update operations
     *
     * @return mixed Operation result or WP_Error
     */

    public function handle_site_api($action, $args, $data)
    {
        switch ($action) {
            case 'reset':
                // Safety check - require confirmation
                if (empty($args['confirm'])) {
                    return new WP_Error('error', 'This command will delete ALL site content. Use --confirm to proceed.');
                }
                // Load necessary classes
                self::load_manager_classes();
                // Parse excluded types
                $excluded = [];
                if ( ! empty($assoc_args['exclude'])) {
                    $excluded = array_map('trim', explode(',', $assoc_args['exclude']));
                }
                $force = ! empty($args['force']);
                $results = [
                    'posts'      => ['deleted' => 0, 'failed' => 0, 'skipped' => false],
                    'pages'      => ['deleted' => 0, 'failed' => 0, 'skipped' => false],
                    'menus'      => ['deleted' => 0, 'failed' => 0, 'skipped' => false],
                    'categories' => ['deleted' => 0, 'failed' => 0, 'skipped' => false],
                    'tags'       => ['deleted' => 0, 'failed' => 0, 'skipped' => false],
                ];

                try {
                    // Delete Posts
                    if ( ! in_array('posts', $excluded)) {
                        $posts        = get_posts(['post_type' => 'post', 'numberposts' => -1, 'post_status' => 'any']);
                        $post_manager = WVC_Post_Manager::get_instance();
                        foreach ($posts as $post) {
                            $result = $post_manager->delete_post($post->ID, $force);
                            if (is_wp_error($result)) {
                                $results['posts']['failed']++;
                            } else {
                                $results['posts']['deleted']++;
                            }
                        }
                    } else {
                        $results['posts']['skipped'] = true;
                    }
                    // Delete Pages
                    if ( ! in_array('pages', $excluded)) {
                        $pages        = get_posts(['post_type' => 'page', 'numberposts' => -1, 'post_status' => 'any']);
                        $page_manager = WVC_Page_Manager::get_instance();
                        foreach ($pages as $page) {
                            $result = $page_manager->delete_page($page->ID, $force);
                            if (is_wp_error($result)) {
                                $results['pages']['failed']++;
                            } else {
                                $results['pages']['deleted']++;
                            }
                        }
                    } else {
                        $results['pages']['skipped'] = true;
                    }
                    // Delete Menus
                    if ( ! in_array('menus', $excluded)) {
                        $menus        = wp_get_nav_menus();
                        $menu_manager = WVC_Menu_Manager::get_instance();
                        foreach ($menus as $menu) {
                            $result = $menu_manager->delete_menu($menu->term_id);
                            if (is_wp_error($result)) {
                                $results['menus']['failed']++;
                            } else {
                                $results['menus']['deleted']++;
                            }
                        }
                    } else {
                        $results['menus']['skipped'] = true;
                    }

                    // Delete Categories (except "Uncategorized")
                    if ( ! in_array('categories', $excluded)) {
                        $categories   = get_categories(['hide_empty' => false]);
                        $post_manager = WVC_Post_Manager::get_instance();

                        foreach ($categories as $category) {
                            // Skip the default "Uncategorized" category
                            if ($category->slug === 'uncategorized') {
                                continue;
                            }
                            $result = $post_manager->delete_category($category->term_id);
                            if (is_wp_error($result)) {
                                $results['categories']['failed']++;
                            } else {
                                $results['categories']['deleted']++;
                            }
                        }
                    } else {
                        $results['categories']['skipped'] = true;
                    }
                    // Delete Tags
                    if ( ! in_array('tags', $excluded)) {
                        $tags         = get_tags(['hide_empty' => false]);
                        $post_manager = WVC_Post_Manager::get_instance();
                        foreach ($tags as $tag) {
                            $result = $post_manager->delete_tag($tag->term_id);
                            if (is_wp_error($result)) {
                                $results['tags']['failed']++;
                            } else {
                                $results['tags']['deleted']++;
                            }
                        }
                    } else {
                        $results['tags']['skipped'] = true;
                    }

                    WP_CLI::log(json_encode($results, JSON_PRETTY_PRINT));

                } catch (Exception $e) {
                    return new WP_Error('error', 'Site reset failed: ' . $e->getMessage());
                }
            default:
                return new WP_Error('invalid_action', "Unknown action: $action for site resource");
        }
    }


    /**
     * Handle menu API operations
     *
     * @param string $action Action to perform
     * @param array $args Command arguments
     * @param array $data Data for create/update operations
     *
     * @return mixed Operation result or WP_Error
     */
    private function handle_menu_api($action, $args, $data)
    {
        $menu_manager = WVC_Menu_Manager::get_instance();

        switch ($action) {
            case 'list':
                return $menu_manager->get_menus();

            case 'get':
                if (empty($args['id'])) {
                    return new WP_Error('missing_id', 'Menu ID is required for this operation');
                }

                return $menu_manager->get_menu($args['id']);

            case 'create':
                return $menu_manager->create_menu($data);

            case 'bulk-create':
                $menus_data = $this->get_bulk_data($args, 'menus');
                if (is_wp_error($menus_data)) {
                    return $menus_data;
                }
                // Process menus
                $results = [
                    'success' => [],
                    'failed'  => []
                ];
                foreach ($menus_data as $menu_data) {
                    $result = $menu_manager->create_menu($menu_data);

                    if (is_wp_error($result)) {
                        $results['failed'][] = [
                            'name'  => isset($menu_data['name']) ? $menu_data['name'] : 'Unknown',
                            'error' => $result->get_error_message()
                        ];
                    } else {
                        $results['success'][] = [
                            'id'          => $result['id'],
                            'name'        => $result['name'],
                            'location'    => isset($result['location']) ? $result['location'] : 'none',
                            'items_count' => isset($result['items']) ? count($result['items']) : 0
                        ];
                    }
                }
                if (empty( $results['success'])){
                    return  new WP_Error('error', 'Failed to create items. '.json_encode($results['failed']));
                }
                return $results['success'];

            case 'delete':
                if (empty($args['id'])) {
                    return new WP_Error('missing_id', 'Menu ID is required for this operation');
                }

                return $menu_manager->delete_menu($args['id']);

            case 'create-item':
                if (empty($args['id'])) {
                    return new WP_Error('missing_id', 'Menu ID is required for this operation');
                }

                $menu_item_id = $menu_manager->add_menu_item($args['id'], $data);
                $menu_item = get_post($menu_item_id);
                $item_data = $menu_manager->prepare_menu_item_response($menu_item);
                return [
                    'success' => true,
                    'menu_id' => $args['id'],
                    'item'    => $item_data,
                ];

            case 'bulk-create-item':
                if (empty($args['id'])) {
                    return new WP_Error('missing_id', 'Menu ID is required for this operation');
                }

                $menu_id    = $args['id'];
                $items_data = $this->get_bulk_data($args, 'menu items');
                if (is_wp_error($items_data)) {
                    return $items_data;
                }
                // Process menu items
                $results = [
                    'success' => [],
                    'failed'  => []
                ];

                foreach ($items_data as $item_data) {
                    $item_id = $menu_manager->add_menu_item($menu_id, $item_data);

                    if (is_wp_error($item_id)) {
                        $results['failed'][] = [
                            'title' => isset($item_data['title']) ? $item_data['title'] : 'Unknown',
                            'error' => $item_id->get_error_message()
                        ];
                    } else {
                        $results['success'][] = [
                            'id'        => $item_id,
                            'title'     => $item_data['title'],
                            'url'       => isset($item_data['title']['url']) ? $item_data['title']['url'] : '',
                            'object_id' => isset($item_data["object_id"]) ? $item_data["object_id"] : 0,
                            'slug'      => isset($item_data["slug"]) ? $item_data["slug"] : '',
                        ];
                    }
                }
                if (empty( $results['success'])){
                    return  new WP_Error('error', 'Failed to create items. '.json_encode($results['failed']));
                }
                return $results['success'];

            case 'update-item':
                if (empty($args['id']) || empty($args['item_id'])) {
                    return new WP_Error('missing_id', 'Menu ID and item ID are required for this operation');
                }
                return $menu_manager->edit_menu_item($args['id'], $args['item_id'], $data);

            case 'delete-item':
                if (empty($args['id']) || empty($args['item_id'])) {
                    return new WP_Error('missing_id', 'Menu ID and item ID are required for this operation');
                }

                return $menu_manager->delete_menu_item($args['id'], $args['item_id']);

            case 'bulk-delete':
                $menus_data = $this->get_bulk_data($args, 'menu IDs');
                if (is_wp_error($menus_data)) {
                    return $menus_data;
                }

                $results = [
                    'success' => [],
                    'failed'  => []
                ];

                foreach ($menus_data as $menu_data) {
                    $menu_id = isset($menu_data['id']) ? $menu_data['id'] : $menu_data;
                    $result = $menu_manager->delete_menu($menu_id);

                    if (is_wp_error($result)) {
                        $results['failed'][] = [
                            'id'    => $menu_id,
                            'error' => $result->get_error_message()
                        ];
                    } else {
                        $results['success'][] = [
                            'id'      => $menu_id,
                            'message' => $result['message']
                        ];
                    }
                }

                if (empty($results['success'])) {
                    return new WP_Error('error', 'Failed to delete any menus. ' . json_encode($results['failed']));
                }
                return $results;

            case 'bulk-delete-item':
                if (empty($args['id'])) {
                    return new WP_Error('missing_id', 'Menu ID is required for this operation');
                }

                $menu_id = $args['id'];
                $items_data = $this->get_bulk_data($args, 'menu item IDs');
                if (is_wp_error($items_data)) {
                    return $items_data;
                }

                $results = [
                    'success' => [],
                    'failed'  => []
                ];

                foreach ($items_data as $item_data) {
                    $item_id = isset($item_data['item_id']) ? $item_data['item_id'] : $item_data;
                    $result = $menu_manager->delete_menu_item($menu_id, $item_id);

                    if (is_wp_error($result)) {
                        $results['failed'][] = [
                            'item_id' => $item_id,
                            'error'   => $result->get_error_message()
                        ];
                    } else {
                        $results['success'][] = [
                            'item_id' => $item_id,
                            'message' => $result['message']
                        ];
                    }
                }

                if (empty($results['success'])) {
                    return new WP_Error('error', 'Failed to delete any menu items. ' . json_encode($results['failed']));
                }
                return $results;

            default:
                return new WP_Error('invalid_action', "Unknown action: $action for menu resource");
        }
    }

    /**
     * Handle page API operations
     *
     * @param string $action Action to perform
     * @param array $args Command arguments
     * @param array $data Data for create/update operations
     *
     * @return mixed Operation result or WP_Error
     */
    private function handle_page_api($action, $args, $data)
    {
        $page_manager = WVC_Page_Manager::get_instance();

        switch ($action) {
            case 'get':
                if (empty($args['id'])) {
                    return new WP_Error('missing_id', 'Page ID is required for this operation');
                }

                return $page_manager->get_page($args['id']);

            case 'create':
                return $page_manager->create_page($data);

            case 'bulk-create':
                $pages_data = $this->get_bulk_data($args, 'pages');
                if (is_wp_error($pages_data)) {
                    return $pages_data;
                }
                // Process pages
                $results = [
                    'success' => [],
                    'failed'  => []
                ];
                foreach ($pages_data as $page_data) {
                    $result = $page_manager->create_page($page_data);

                    if (is_wp_error($result)) {
                        $results['failed'][] = [
                            'title' => isset($page_data['title']) ? $page_data['title'] : 'Unknown',
                            'error' => $result->get_error_message()
                        ];
                    } else {
                        $success = [
                            'id'      => $result['id'],
                            'title'   => $result['title'],
                            'page_id' => $result['id'],
                        ];
                        if (isset($page_data["_item"])) {
                            $success["slug"]           = $page_data["_item"]["slug"];
                            $success["component_name"] = $page_data["_item"]["component_name"];
                        }
                        $results['success'][] = $success;

                    }
                }
                if (empty( $results['success'])){
                    return  new WP_Error('error', 'Failed to create items. '.json_encode($results['failed']));
                }
                return $results['success'];

            case 'update':
                if (empty($args['id'])) {
                    return new WP_Error('missing_id', 'Page ID is required for this operation');
                }

                return $page_manager->edit_page($args['id'], $data);

            case 'delete':
                if (empty($args['id'])) {
                    return new WP_Error('missing_id', 'Page ID is required for this operation');
                }
                $force = ! empty($args['force']) && $args['force'] === 'true';

                return $page_manager->delete_page($args['id'], $force);

            case 'bulk-delete':
                $pages_data = $this->get_bulk_data($args, 'page IDs');
                if (is_wp_error($pages_data)) {
                    return $pages_data;
                }

                $force = ! empty($args['force']) && $args['force'] === 'true';
                $results = [
                    'success' => [],
                    'failed'  => []
                ];

                foreach ($pages_data as $page_data) {
                    $page_id = isset($page_data['id']) ? $page_data['id'] : $page_data;
                    $result = $page_manager->delete_page($page_id, $force);

                    if (is_wp_error($result)) {
                        $results['failed'][] = [
                            'id'    => $page_id,
                            'error' => $result->get_error_message()
                        ];
                    } else {
                        $results['success'][] = [
                            'id'      => $page_id,
                            'message' => $result['message'] ?? 'Page deleted successfully'
                        ];
                    }
                }

                if (empty($results['success'])) {
                    return new WP_Error('error', 'Failed to delete any pages. ' . json_encode($results['failed']));
                }
                return $results;

            default:
                return new WP_Error('invalid_action', "Unknown action: $action for page resource");
        }
    }

    /**
     * Handle post API operations
     *
     * @param string $action Action to perform
     * @param array $args Command arguments
     * @param array $data Data for create/update operations
     *
     * @return mixed Operation result or WP_Error
     */
    private function handle_post_api($action, $args, $data)
    {
        $post_manager = WVC_Post_Manager::get_instance();

        switch ($action) {
            case 'get':
                if (empty($args['id'])) {
                    return new WP_Error('missing_id', 'Post ID is required for this operation');
                }

                return $post_manager->get_post($args['id']);

            case 'create':
                return $post_manager->create_post($data);

            case 'bulk-create':
                $posts_data = $this->get_bulk_data($args, 'posts');
                if (is_wp_error($posts_data)) {
                    return $posts_data;
                }

                // Process posts
                $results = [
                    'success' => [],
                    'failed'  => []
                ];

                foreach ($posts_data as $post_data) {
                    $result = $post_manager->create_post($post_data);

                    if (is_wp_error($result)) {
                        $results['failed'][] = [
                            'title' => isset($post_data['title']) ? $post_data['title'] : 'Unknown',
                            'error' => $result->get_error_message()
                        ];
                    } else {
                        $results['success'][] = [
                            'id'        => $result['id'],
                            'title'     => $result['title'],
                            'permalink' => $result['permalink']
                        ];
                    }
                }
                if (empty( $results['success'])){
                    return  new WP_Error('error', 'Failed to create items. '.json_encode($results['failed']));
                }
                return  $results['success'];

            case 'update':
                if (empty($args['id'])) {
                    return new WP_Error('missing_id', 'Post ID is required for this operation');
                }

                return $post_manager->update_post($args['id'], $data);

            case 'delete':
                if (empty($args['id'])) {
                    return new WP_Error('missing_id', 'Post ID is required for this operation');
                }
                $force = ! empty($args['force']) && $args['force'] === 'true';

                return $post_manager->delete_post($args['id'], $force);

            case 'bulk-delete':
                $posts_data = $this->get_bulk_data($args, 'post IDs');
                if (is_wp_error($posts_data)) {
                    return $posts_data;
                }

                $force = ! empty($args['force']) && $args['force'] === 'true';
                $results = [
                    'success' => [],
                    'failed'  => []
                ];

                foreach ($posts_data as $post_data) {
                    $post_id = isset($post_data['id']) ? $post_data['id'] : $post_data;
                    $result = $post_manager->delete_post($post_id, $force);

                    if (is_wp_error($result)) {
                        $results['failed'][] = [
                            'id'    => $post_id,
                            'error' => $result->get_error_message()
                        ];
                    } else {
                        $results['success'][] = [
                            'id'      => $post_id,
                            'message' => $result['message'] ?? 'Post deleted successfully'
                        ];
                    }
                }

                if (empty($results['success'])) {
                    return new WP_Error('error', 'Failed to delete any posts. ' . json_encode($results['failed']));
                }
                return $results;

            default:
                return new WP_Error('invalid_action', "Unknown action: $action for post resource");
        }
    }

    /**
     * Handle category API operations
     *
     * @param string $action Action to perform
     * @param array $args Command arguments
     * @param array $data Data for create/update operations
     *
     * @return mixed Operation result or WP_Error
     */
    private function handle_category_api($action, $args, $data)
    {
        $post_manager = WVC_Post_Manager::get_instance();

        switch ($action) {
            case 'get':
                if (empty($args['id'])) {
                    return new WP_Error('missing_id', 'Category ID is required for this operation');
                }

                return $post_manager->get_category($args['id']);

            case 'create':
                return $post_manager->create_category($data);

            case 'bulk-create':
                $categories_data = $this->get_bulk_data($args, 'categories');
                if (is_wp_error($categories_data)) {
                    return $categories_data;
                }

                // Process categories
                $results = [
                    'success' => [],
                    'failed'  => []
                ];

                foreach ($categories_data as $category_data) {
                    $result = $post_manager->create_category($category_data);

                    if (is_wp_error($result)) {
                        $results['failed'][] = [
                            'name'  => isset($category_data['name']) ? $category_data['name'] : 'Unknown',
                            'error' => $result->get_error_message()
                        ];
                    } else {
                        $results['success'][] = [
                            'id'     => $result['id'],
                            'name'   => $result['name'],
                            'slug'   => $result['slug'],
                            'parent' => isset($result['parent']) ? $result['parent'] : 0
                        ];
                    }
                }
                if (empty( $results['success'])){
                    return  new WP_Error('error', 'Failed to create items. '.json_encode($results['failed']));
                }
                return $results['success'];

            case 'update':
                if (empty($args['id'])) {
                    return new WP_Error('missing_id', 'Category ID is required for this operation');
                }

                return $post_manager->update_category($args['id'], $data);

            case 'delete':
                if (empty($args['id'])) {
                    return new WP_Error('missing_id', 'Category ID is required for this operation');
                }

                return $post_manager->delete_category($args['id']);

            case 'bulk-delete':
                $categories_data = $this->get_bulk_data($args, 'category IDs');
                if (is_wp_error($categories_data)) {
                    return $categories_data;
                }

                $results = [
                    'success' => [],
                    'failed'  => []
                ];

                foreach ($categories_data as $category_data) {
                    $category_id = isset($category_data['id']) ? $category_data['id'] : $category_data;
                    $result = $post_manager->delete_category($category_id);

                    if (is_wp_error($result)) {
                        $results['failed'][] = [
                            'id'    => $category_id,
                            'error' => $result->get_error_message()
                        ];
                    } else {
                        $results['success'][] = [
                            'id'      => $category_id,
                            'message' => $result['message'] ?? 'Category deleted successfully'
                        ];
                    }
                }

                if (empty($results['success'])) {
                    return new WP_Error('error', 'Failed to delete any categories. ' . json_encode($results['failed']));
                }
                return $results;

            default:
                return new WP_Error('invalid_action', "Unknown action: $action for category resource");
        }
    }

    /**
     * Handle tag API operations
     *
     * @param string $action Action to perform
     * @param array $args Command arguments
     * @param array $data Data for create/update operations
     *
     * @return mixed Operation result or WP_Error
     */
    private function handle_tag_api($action, $args, $data)
    {
        $post_manager = WVC_Post_Manager::get_instance();

        switch ($action) {
            case 'get':
                if (empty($args['id'])) {
                    return new WP_Error('missing_id', 'Tag ID is required for this operation');
                }

                return $post_manager->get_tag($args['id']);

            case 'create':
                return $post_manager->create_tag($data);

            case 'bulk-create':
                $tags_data = $this->get_bulk_data($args, 'tags');
                if (is_wp_error($tags_data)) {
                    return $tags_data;
                }

                // Process tags
                $results = [
                    'success' => [],
                    'failed'  => []
                ];

                foreach ($tags_data as $tag_data) {
                    $result = $post_manager->create_tag($tag_data);

                    if (is_wp_error($result)) {
                        $results['failed'][] = [
                            'name'  => isset($tag_data['name']) ? $tag_data['name'] : 'Unknown',
                            'error' => $result->get_error_message()
                        ];
                    } else {
                        $results['success'][] = [
                            'id'   => $result['id'],
                            'name' => $result['name'],
                            'slug' => $result['slug']
                        ];
                    }
                }
                if (empty( $results['success'])){
                    return  new WP_Error('error', 'Failed to create items. '.json_encode($results['failed']));
                }
                return $results['success'];

            case 'update':
                if (empty($args['id'])) {
                    return new WP_Error('missing_id', 'Tag ID is required for this operation');
                }

                return $post_manager->update_tag($args['id'], $data);

            case 'delete':
                if (empty($args['id'])) {
                    return new WP_Error('missing_id', 'Tag ID is required for this operation');
                }

                return $post_manager->delete_tag($args['id']);

            case 'bulk-delete':
                $tags_data = $this->get_bulk_data($args, 'tag IDs');
                if (is_wp_error($tags_data)) {
                    return $tags_data;
                }

                $results = [
                    'success' => [],
                    'failed'  => []
                ];

                foreach ($tags_data as $tag_data) {
                    $tag_id = isset($tag_data['id']) ? $tag_data['id'] : $tag_data;
                    $result = $post_manager->delete_tag($tag_id);

                    if (is_wp_error($result)) {
                        $results['failed'][] = [
                            'id'    => $tag_id,
                            'error' => $result->get_error_message()
                        ];
                    } else {
                        $results['success'][] = [
                            'id'      => $tag_id,
                            'message' => $result['message'] ?? 'Tag deleted successfully'
                        ];
                    }
                }

                if (empty($results['success'])) {
                    return new WP_Error('error', 'Failed to delete any tags. ' . json_encode($results['failed']));
                }
                return $results;

            default:
                return new WP_Error('invalid_action', "Unknown action: $action for tag resource");
        }
    }

    /**
     * Handle media API operations
     *
     * @param string $action Action to perform
     * @param array $args Command arguments
     * @param array $data Data for create/update operations
     *
     * @return mixed Operation result or WP_Error
     */
    private function handle_media_api($action, $args, $data)
    {
        $post_manager = WVC_Post_Manager::get_instance();

        switch ($action) {
            case 'get':
                if (empty($args['id'])) {
                    return new WP_Error('missing_id', 'Media ID is required for this operation');
                }

                return $post_manager->get_media($args['id']);

            case 'upload':
                // Check if we have base64 data or file path
                if (empty($data['base64_data']) && empty($data['file'])) {
                    return new WP_Error('missing_data', 'Either base64_data or file path is required for media upload');
                }

                $post_id = isset($data['post_id']) ? $data['post_id'] : 0;

                // Set up arguments for media upload
                $upload_args = array(
                    'title'       => isset($data['title']) ? $data['title'] : '',
                    'caption'     => isset($data['caption']) ? $data['caption'] : '',
                    'description' => isset($data['description']) ? $data['description'] : '',
                    'alt_text'    => isset($data['alt_text']) ? $data['alt_text'] : '',
                );

                // If base64 data is provided, use it
                if ( ! empty($data['base64_data'])) {
                    // Extract the data from the base64 string
                    $base64_data = $data['base64_data'];

                    // Check if it's a data URL (e.g., data:image/jpeg;base64,/9j/4AAQ...)
                    if (strpos($base64_data, 'data:') === 0) {
                        // Parse the content type and data
                        list($header, $base64_data) = explode(',', $base64_data, 2);
                        preg_match('/data:([^;]+)/', $header, $matches);
                        $content_type = ! empty($matches[1]) ? $matches[1] : 'application/octet-stream';
                    } else {
                        // If no data URL format, assume the content type from the filename or default
                        $content_type = isset($data['content_type']) ? $data['content_type'] : 'application/octet-stream';
                    }

                    // Decode the base64 data
                    $file_data = base64_decode($base64_data);

                    // Make sure we have a valid filename
                    if (empty($data['filename'])) {
                        // Generate a filename based on content type
                        switch ($content_type) {
                            case 'image/jpeg':
                                $extension = 'jpg';
                                break;
                            case 'image/png':
                                $extension = 'png';
                                break;
                            case 'image/gif':
                                $extension = 'gif';
                                break;
                            case 'image/svg+xml':
                                $extension = 'svg';
                                break;
                            case 'image/webp':
                                $extension = 'webp';
                                break;
                            default:
                                $extension = 'bin';
                                break;
                        }
                        $filename = 'upload-' . time() . '.' . $extension;
                    } else {
                        $filename = $data['filename'];
                    }
                } else {
                    // Fall back to file path method
                    $file_path = $data['file'];
                    if ( ! file_exists($file_path) || ! is_readable($file_path)) {
                        return new WP_Error('invalid_file', 'File does not exist or is not readable');
                    }

                    $file_data    = file_get_contents($file_path);
                    $filename     = basename($file_path);
                    $content_type = mime_content_type($file_path);
                }

                // Use the upload_media method
                return $post_manager->upload_media(
                    $file_data,
                    $filename,
                    $content_type,
                    $post_id,
                    $upload_args
                );

            case 'update':
                if (empty($args['id'])) {
                    return new WP_Error('missing_id', 'Media ID is required for this operation');
                }

                return $post_manager->update_media($args['id'], $data);

            case 'delete':
                if (empty($args['id'])) {
                    return new WP_Error('missing_id', 'Media ID is required for this operation');
                }
                $force = ! empty($args['force']) && $args['force'] === 'true';

                return $post_manager->delete_media($args['id'], $force);

            default:
                return new WP_Error('invalid_action', "Unknown action: $action for media resource");
        }
    }

    /**
     * Handle settings API operations
     *
     * @param string $action Action to perform
     * @param array $args Command arguments
     * @param array $data Data for update operations
     *
     * @return mixed Operation result or WP_Error
     */
    private function handle_settings_api($action, $args, $data)
    {
        // Create a temporary instance of the settings REST API class
        require_once get_template_directory() . '/includes/rest/wvc-settings-rest-api.php';
        $settings_api = new WVC_Settings_REST_API();

        switch ($action) {
            case 'get':
                // Call the private method directly with reflection
                $reflection = new ReflectionObject($settings_api);
                $method     = $reflection->getMethod('get_general_settings');
                $method->setAccessible(true);

                return $method->invoke($settings_api);

            case 'update':
                if (empty($data)) {
                    return new WP_Error('missing_data', 'Settings data is required for update operation');
                }

                // Create a mock request object to pass to the update_settings method
                $request = new WP_REST_Request('PUT', '/wvc/v1/settings');
                $request->set_param('settings', $data);

                // Call the update_settings method
                return $settings_api->update_settings($request);

            default:
                return new WP_Error('invalid_action', "Unknown action: $action for settings resource");
        }
    }

    /**
     * Handle delete-restore API operations
     *
     * @param string $action Action to perform
     * @param array $args Command arguments
     * @param array $data Data for the operation
     *
     * @return mixed Operation result or WP_Error
     */
    private function handle_delete_restore_api($action, $args, $data)
    {
        // Include necessary files
        require_once get_template_directory() . '/includes/delete_restore/post.php';

        switch ($action) {
            case 'delete':
                if (empty($data['type'])) {
                    return new WP_Error('missing_type', 'Item type is required (post or term)');
                }

                if (empty($data['id'])) {
                    return new WP_Error('missing_id', 'Item ID is required');
                }

                $type = $data['type'];
                $id   = $data['id'];

                if ($type === 'post') {
                    $result = wvc_delete_post($id);
                    if (is_wp_error($result)) {
                        return $result;
                    }

                    return [
                        'message' => 'Post deleted successfully',
                        'id'      => $id
                    ];
                } elseif ($type === 'term') {
                    return new WP_Error('not_implemented', 'Term deletion not implemented yet');
                } else {
                    return new WP_Error('invalid_type', 'Invalid type parameter. Must be "post" or "term"');
                }

            case 'restore':
                if (empty($data['type'])) {
                    return new WP_Error('missing_type', 'Item type is required (post or term)');
                }

                if (empty($data['id'])) {
                    return new WP_Error('missing_id', 'Item ID is required');
                }

                $type = $data['type'];
                $id   = $data['id'];

                if ($type === 'post') {
                    $result = wvc_restore_post($id);
                    if (is_wp_error($result)) {
                        return $result;
                    }

                    return [
                        'message' => 'Post restored successfully',
                        'id'      => $id
                    ];
                } elseif ($type === 'term') {
                    return new WP_Error('not_implemented', 'Term restoration not implemented yet');
                } else {
                    return new WP_Error('invalid_type', 'Invalid type parameter. Must be "post" or "term"');
                }

            case 'bulk-delete':
                $items_data = $this->get_bulk_data($args, 'items');
                if (is_wp_error($items_data)) {
                    return $items_data;
                }

                $results = [
                    'success' => [],
                    'failed'  => []
                ];

                foreach ($items_data as $item_data) {
                    if (empty($item_data['type']) || empty($item_data['id'])) {
                        $results['failed'][] = [
                            'item'  => $item_data,
                            'error' => 'Missing type or id parameter'
                        ];
                        continue;
                    }

                    $type = $item_data['type'];
                    $id   = $item_data['id'];

                    if ($type === 'post') {
                        $result = wvc_delete_post($id);
                        if (is_wp_error($result)) {
                            $results['failed'][] = [
                                'type'  => $type,
                                'id'    => $id,
                                'error' => $result->get_error_message()
                            ];
                        } else {
                            $results['success'][] = [
                                'type'    => $type,
                                'id'      => $id,
                                'message' => 'Post deleted successfully'
                            ];
                        }
                    } elseif ($type === 'term') {
                        $results['failed'][] = [
                            'type'  => $type,
                            'id'    => $id,
                            'error' => 'Term deletion not implemented yet'
                        ];
                    } else {
                        $results['failed'][] = [
                            'type'  => $type,
                            'id'    => $id,
                            'error' => 'Invalid type parameter. Must be "post" or "term"'
                        ];
                    }
                }

                if (empty($results['success'])) {
                    return new WP_Error('error', 'Failed to delete any items. ' . json_encode($results['failed']));
                }
                return $results;

            case 'bulk-restore':
                $items_data = $this->get_bulk_data($args, 'items');
                if (is_wp_error($items_data)) {
                    return $items_data;
                }

                $results = [
                    'success' => [],
                    'failed'  => []
                ];

                foreach ($items_data as $item_data) {
                    if (empty($item_data['type']) || empty($item_data['id'])) {
                        $results['failed'][] = [
                            'item'  => $item_data,
                            'error' => 'Missing type or id parameter'
                        ];
                        continue;
                    }

                    $type = $item_data['type'];
                    $id   = $item_data['id'];

                    if ($type === 'post') {
                        $result = wvc_restore_post($id);
                        if (is_wp_error($result)) {
                            $results['failed'][] = [
                                'type'  => $type,
                                'id'    => $id,
                                'error' => $result->get_error_message()
                            ];
                        } else {
                            $results['success'][] = [
                                'type'    => $type,
                                'id'      => $id,
                                'message' => 'Post restored successfully'
                            ];
                        }
                    } elseif ($type === 'term') {
                        $results['failed'][] = [
                            'type'  => $type,
                            'id'    => $id,
                            'error' => 'Term restoration not implemented yet'
                        ];
                    } else {
                        $results['failed'][] = [
                            'type'  => $type,
                            'id'    => $id,
                            'error' => 'Invalid type parameter. Must be "post" or "term"'
                        ];
                    }
                }

                if (empty($results['success'])) {
                    return new WP_Error('error', 'Failed to restore any items. ' . json_encode($results['failed']));
                }
                return $results;

            default:
                return new WP_Error('invalid_action', "Unknown action: $action for delete-restore resource");
        }
    }

    /**
     * Flatten an array for table output
     *
     * @param array $array The array to flatten
     * @param string $prefix Prefix for nested keys
     *
     * @return array Flattened array
     */
    private function flatten_array($array, $prefix = '')
    {
        $result = [];

        foreach ($array as $key => $value) {
            $new_key = $prefix ? $prefix . '.' . $key : $key;

            if (is_array($value) && ! $this->is_sequential_array($value)) {
                $result = array_merge($result, $this->flatten_array($value, $new_key));
            } else {
                $result[$new_key] = is_array($value) ? json_encode($value) : $value;
            }
        }

        return $result;
    }

    /**
     * Check if an array is a sequential array (not associative)
     *
     * @param array $array The array to check
     *
     * @return bool True if sequential, false if associative
     */
    private function is_sequential_array($array)
    {
        if ( ! is_array($array)) {
            return false;
        }

        return array_keys($array) === range(0, count($array) - 1);
    }

    /**
     * Get bulk data from file or JSON string
     *
     * @param array $args Command arguments
     * @param string $type Type of data (for error messages)
     *
     * @return array|WP_Error Array of data or WP_Error on failure
     */
    private function get_bulk_data($args, $type)
    {
        // Check if we have a file or JSON data
        if ( ! empty($args['file'])) {
            $file_path = $args['file'];

            if ( ! file_exists($file_path)) {
                return new WP_Error('missing_file', "File does not exist: $file_path");
            }

            if ( ! is_readable($file_path)) {
                return new WP_Error('unreadable_file', "File is not readable: $file_path");
            }

            $json_data = file_get_contents($file_path);
            $data      = json_decode($json_data, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                return new WP_Error('invalid_json', "Invalid JSON in file: " . json_last_error_msg());
            }
        } elseif ( ! empty($args['data'])) {
            $json_data = $args['data'];
            $data      = json_decode($json_data, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                return new WP_Error('invalid_json', "Invalid JSON: " . json_last_error_msg());
            }
        } else {
            return new WP_Error('missing_data', "Either --file or --data argument is required for bulk operations");
        }

        // Validate the data
        if ( ! is_array($data)) {
            return new WP_Error('invalid_data', "Data must be an array of $type");
        }

        if (empty($data)) {
            return new WP_Error('empty_data', "No $type data found in the provided JSON");
        }

        // Check if it's an array of objects
        if ( ! $this->is_sequential_array($data)) {
            // Single object was provided, convert to array
            $data = [$data];
        }

        return $data;
    }
}