<?php
/**
 * Style Kit Schema Validator
 *
 * This class provides validation and schema management for Style Kit
 * configurations. It validates CSS variables, schema structure, and
 * ensures data integrity for Style Kit JSON configurations.
 *
 * @package    WVC_Theme
 * @subpackage Style_Kits
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 * @link       https://github.com/your-repo/wvc-theme
 */

if (!defined("ABSPATH")) {
    exit;
}

class WVC_Style_Kit_Schema
{
    
    private static $schema = null;
    private static $schema_file = null;
    
    /**
     * Initialize the schema file path
     */
    public static function init()
    {
        self::$schema_file = get_template_directory() . "/includes/kits/style-kit-schema.json";
    }
    
    /**
     * Load and parse JSON schema file
     */
    public static function load_schema()
    {
        if (self::$schema !== null) {
            return self::$schema;
        }
        
        if (self::$schema_file === null) {
            self::init();
        }
        
        if (!file_exists(self::$schema_file)) {
            return new WP_Error("schema_not_found", __("Schema file not found.", "wvc-theme"));
        }
        
        $json_content = file_get_contents(self::$schema_file);
        if ($json_content === false) {
            return new WP_Error("schema_read_error", __("Could not read schema file.", "wvc-theme"));
        }
        
        // Parse JSON content-managers
        $parsed_schema = self::parse_json($json_content);
        if (is_wp_error($parsed_schema)) {
            return $parsed_schema;
        }
        
        self::$schema = $parsed_schema;
        return self::$schema;
    }
    
    /**
     * JSON parser for our schema structure
     */
    private static function parse_json($json_content)
    {
        // Decode JSON content-managers
        $decoded = json_decode($json_content, true);
        
        // Check for JSON errors
        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_Error("json_parse_error", __("Invalid JSON format: " . json_last_error_msg(), "wvc-theme"));
        }
        
        // Validate that we have the expected structure
        if (!is_array($decoded)) {
            return new WP_Error("json_structure_error", __("JSON schema must be an object.", "wvc-theme"));
        }
        
        return $decoded;
    }
    
    /**
     * Get all CSS variables from schema
     */
    public static function get_css_variables()
    {
        $schema = self::load_schema();
        if (is_wp_error($schema)) {
            return $schema;
        }
        
        return isset($schema["css_variables"]) ? $schema["css_variables"] : array();
    }
    
    /**
     * Get style kit metadata from schema
     */
    public static function get_style_kit_metadata()
    {
        $schema = self::load_schema();
        if (is_wp_error($schema)) {
            return $schema;
        }
        
        return isset($schema["style_kit"]) ? $schema["style_kit"] : array();
    }
    
    /**
     * Get allowed variable types
     */
    public static function get_allowed_types()
    {
        return array(
            "color", 
            "font-family", 
            "size", 
            "number", 
            "string", 
            "url", 
            "gradient",
            "time",
            "percentage", 
            "angle",
            "timing-function",
            "shadow",
            "transform",
            "filter",
            "border",
            "keyword"
        );
    }
    
    /**
     * Get allowed categories
     */
    public static function get_allowed_categories()
    {
        $metadata = self::get_style_kit_metadata();
        if (is_wp_error($metadata)) {
            return array("general");
        }
        
        return isset($metadata["categories"]) ? $metadata["categories"] : array("general");
    }
    
    /**
     * Validate CSS variable structure
     */
    public static function validate_css_variable($variable)
    {
        if (!is_array($variable)) {
            return false;
        }
        
        // Required fields
        $required_fields = array("key", "value", "type");
        foreach ($required_fields as $field) {
            if (!isset($variable[$field]) || empty($variable[$field])) {
                return false;
            }
        }
        
        // Validate type
        $allowed_types = self::get_allowed_types();
        if (!in_array($variable["type"], $allowed_types)) {
            return false;
        }
        
        // Validate category if provided
        if (isset($variable["category"]) && !empty($variable["category"])) {
            $allowed_categories = self::get_allowed_categories();
            if (!in_array($variable["category"], $allowed_categories)) {
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * Sanitize CSS variable
     */
    public static function sanitize_css_variable($variable)
    {
        $sanitized = array(
            "key" => sanitize_key($variable["key"]),
            "value" => self::sanitize_css_value($variable["value"], $variable["type"]),
            "type" => sanitize_text_field($variable["type"]),
            "category" => isset($variable["category"]) ? sanitize_text_field($variable["category"]) : "general",
            "description" => isset($variable["description"]) ? sanitize_text_field($variable["description"]) : "",
            "default_value" => isset($variable["default_value"]) ? self::sanitize_css_value($variable["default_value"], $variable["type"]) : ""
        );
        
        return $sanitized;
    }
    
    /**
     * Sanitize CSS value based on type
     */
    public static function sanitize_css_value($value, $type)
    {
        switch ($type) {
        case "color":
            // Handle hex, rgb, rgba, hsl, hsla, named colors
            if (preg_match("/^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3})$/", $value)) {
                return $value; // Valid hex color
            }
            if (preg_match("/^rgba?\([\d\s,%.]+\)$/", $value)) {
                return $value; // Valid rgb/rgba
            }
            if (preg_match("/^hsla?\([\d\s,%.]+\)$/", $value)) {
                return $value; // Valid hsl/hsla
            }
            return sanitize_text_field($value); // For named colors or custom values
                
        case "font-family":
            return sanitize_text_field($value);
                
        case "size":
            // Handle px, em, rem, %, vh, vw, etc.
            if (preg_match("/^[\d.]+(?:px|em|rem|%|vh|vw|vmin|vmax|ch|ex|in|cm|mm|pt|pc)$/", $value)) {
                return $value;
            }
            return "0px"; // Default fallback
                
        case "number":
            return is_numeric($value) ? (string)$value : "0";
                
        case "url":
            return esc_url_raw($value);
                
        case "gradient":
            return sanitize_text_field($value);

        case "time":
            // Handle seconds (s) and milliseconds (ms)
            if (preg_match("/^[\d.]+(?:s|ms)$/", $value)) {
                return $value;
            }
            return "0s"; // Default fallback

        case "percentage":
            // Handle percentage values
            if (preg_match("/^[\d.]+%$/", $value)) {
                return $value;
            }
            return "0%"; // Default fallback

        case "angle":
            // Handle degrees, radians, gradians, turns
            if (preg_match("/^[\d.-]+(?:deg|rad|grad|turn)$/", $value)) {
                return $value;
            }
            return "0deg"; // Default fallback

        case "timing-function":
            // Handle easing functions
            $valid_keywords = array("ease", "ease-in", "ease-out", "ease-in-out", "linear", "step-start", "step-end");
            if (in_array($value, $valid_keywords)) {
                return $value;
            }
            // Handle cubic-bezier functions
            if (preg_match("/^cubic-bezier\([\d\s,.-]+\)$/", $value)) {
                return $value;
            }
            // Handle steps functions
            if (preg_match("/^steps\(\d+(?:,\s*(?:start|end))?\)$/", $value)) {
                return $value;
            }
            return "ease"; // Default fallback

        case "shadow":
            // Handle box-shadow and text-shadow values (complex validation)
            // Basic validation for shadow syntax
            if (preg_match("/^(?:inset\s+)?[\d.-]+(?:px|em|rem)\s+[\d.-]+(?:px|em|rem)(?:\s+[\d.-]+(?:px|em|rem))?(?:\s+[\d.-]+(?:px|em|rem))?(?:\s+#[A-Fa-f0-9]{3,8}|\s+rgba?\([^)]+\)|\s+hsla?\([^)]+\))?$/", $value)) {
                return sanitize_text_field($value);
            }
            if ($value === "none") {
                return $value;
            }
            return "none"; // Default fallback

        case "transform":
            // Handle transform functions
            $valid_functions = array("translate", "translateX", "translateY", "translateZ", "translate3d", "scale", "scaleX", "scaleY", "scaleZ", "scale3d", "rotate", "rotateX", "rotateY", "rotateZ", "rotate3d", "skew", "skewX", "skewY", "matrix", "matrix3d", "perspective");
            $function_pattern = "(?:" . implode("|", $valid_functions) . ")\([^)]+\)";
            if (preg_match("/^(?:\s*" . $function_pattern . "\s*)+$/", $value) || $value === "none") {
                return sanitize_text_field($value);
            }
            return "none"; // Default fallback

        case "filter":
            // Handle filter functions
            $valid_functions = array("blur", "brightness", "contrast", "drop-shadow", "grayscale", "hue-rotate", "invert", "opacity", "saturate", "sepia");
            $function_pattern = "(?:" . implode("|", $valid_functions) . ")\([^)]+\)";
            if (preg_match("/^(?:\s*" . $function_pattern . "\s*)+$/", $value) || $value === "none") {
                return sanitize_text_field($value);
            }
            return "none"; // Default fallback

        case "border":
            // Handle border shorthand: width style color
            if (preg_match("/^[\d.]+(?:px|em|rem|%)\s+(?:solid|dashed|dotted|double|groove|ridge|inset|outset|none|hidden)\s+(?:#[A-Fa-f0-9]{3,8}|rgba?\([^)]+\)|hsla?\([^)]+\)|\w+)$/", $value)) {
                return sanitize_text_field($value);
            }
            if ($value === "none") {
                return $value;
            }
            return "none"; // Default fallback

        case "keyword":
            // Handle CSS keywords - sanitize but allow common keywords
            $common_keywords = array("auto", "inherit", "initial", "unset", "none", "normal", "block", "inline", "inline-block", "flex", "grid", "absolute", "relative", "fixed", "static", "sticky", "hidden", "visible", "transparent", "center", "left", "right", "top", "bottom", "baseline", "stretch", "start", "end", "space-between", "space-around", "space-evenly");
            $clean_value = sanitize_text_field($value);
            return $clean_value;
                
        default:
            return sanitize_text_field($value);
        }
    }
    
    /**
     * Validate and sanitize multiple variables
     */
    public static function validate_and_sanitize_variables($variables)
    {
        if (!is_array($variables)) {
            return new WP_Error("wvc_invalid_variables", __("CSS variables must be an array.", "wvc-theme"), array("status" => 400));
        }
        
        $sanitized = array();
        foreach ($variables as $variable) {
            if (!self::validate_css_variable($variable)) {
                return new WP_Error("wvc_invalid_variable", __("Invalid CSS variable structure.", "wvc-theme"), array("status" => 400));
            }
            $sanitized[] = self::sanitize_css_variable($variable);
        }
        
        return $sanitized;
    }
    
    /**
     * Get variable categories from array of variables
     */
    public static function get_variable_categories($css_variables)
    {
        $categories = array();
        foreach ($css_variables as $variable) {
            $category = isset($variable["category"]) ? $variable["category"] : "general";
            if (!in_array($category, $categories)) {
                $categories[] = $category;
            }
        }
        return $categories;
    }
    
    /**
     * Get default variables from schema
     */
    public static function get_default_variables()
    {
        $schema_variables = self::get_css_variables();
        if (is_wp_error($schema_variables)) {
            return array();
        }
        
        $defaults = array();
        foreach ($schema_variables as $variable) {
            $defaults[] = array(
                "key" => $variable["key"],
                "value" => $variable["default_value"],
                "type" => $variable["type"],
                "category" => $variable["category"],
                "description" => $variable["description"],
                "default_value" => $variable["default_value"]
            );
        }
        
        return $defaults;
    }
    
    /**
     * Validate variable against schema definition
     */
    public static function validate_variable_against_schema($variable)
    {
        $schema_variables = self::get_css_variables();
        if (is_wp_error($schema_variables)) {
            return true; // If schema can't be loaded, skip validation
        }
        
        // Find matching schema variable
        $schema_var = null;
        foreach ($schema_variables as $schema_variable) {
            if ($schema_variable["key"] === $variable["key"]) {
                $schema_var = $schema_variable;
                break;
            }
        }
        
        if (!$schema_var) {
            return true; // Allow custom variables not in schema
        }
        
        // Validate type matches schema
        if ($variable["type"] !== $schema_var["type"]) {
            return new WP_Error("wvc_type_mismatch", sprintf(__("Variable '%s' should be of type '%s', got '%s'.", "wvc-theme"), $variable["key"], $schema_var["type"], $variable["type"]));
        }
        
        return true;
    }
}

// Initialize the schema class
add_action("init", array("WVC_Style_Kit_Schema", "init")); 