<?php
/**
 * Style Kit REST API Controller
 *
 * This class extends the WordPress REST API to provide custom endpoints
 * for Style Kit management. It handles CRUD operations, status management,
 * validation, and permissions for Style Kit posts via REST API.
 *
 * @package    WVC_Theme
 * @subpackage Style_Kits
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 * @link       https://github.com/your-repo/wvc-theme
 */

if (!defined("ABSPATH")) {
    exit;
}

// Load the schema handler
require_once get_template_directory() . "/includes/kits/class-style-kit-schema.php";

class WVC_Style_Kit_REST_API
{
    
    private $namespace = "wvc/v1";
    private $rest_base = "style-kits";
    
    public function __construct()
    {
        add_action("rest_api_init", array($this, "register_routes"));
    }
    
    /**
     * Register REST API routes
     */
    public function register_routes()
    {
        // Get all style kits / Create new style kit
        register_rest_route(
            $this->namespace, "/" . $this->rest_base, array(
            array(
                "methods" => WP_REST_Server::READABLE,
                "callback" => array($this, "get_style_kits"),
                "permission_callback" => array($this, "get_items_permissions_check"),
                "args" => $this->get_collection_params()
            ),
            array(
                "methods" => WP_REST_Server::CREATABLE,
                "callback" => array($this, "create_style_kit"),
                "permission_callback" => array($this, "create_item_permissions_check"),
                "args" => $this->get_endpoint_args_for_item_schema(WP_REST_Server::CREATABLE)
            ),
            "schema" => array($this, "get_public_item_schema")
            )
        );
        
        // Get/Update/Delete specific style kit
        register_rest_route(
            $this->namespace, "/" . $this->rest_base . "/(?P<id>[\d]+)", array(
            array(
                "methods" => WP_REST_Server::READABLE,
                "callback" => array($this, "get_style_kit"),
                "permission_callback" => array($this, "get_item_permissions_check"),
                "args" => array(
                    "context" => $this->get_context_param(array("default" => "view"))
                )
            ),
            array(
                "methods" => WP_REST_Server::EDITABLE,
                "callback" => array($this, "update_style_kit"),
                "permission_callback" => array($this, "update_item_permissions_check"),
                "args" => $this->get_endpoint_args_for_item_schema(WP_REST_Server::EDITABLE)
            ),
            array(
                "methods" => WP_REST_Server::DELETABLE,
                "callback" => array($this, "delete_style_kit"),
                "permission_callback" => array($this, "delete_item_permissions_check")
            ),
            "schema" => array($this, "get_public_item_schema")
            )
        );

        // Get CSS Variables Schema - Static endpoint for AI agents
        register_rest_route(
            $this->namespace, "/" . $this->rest_base . "/schema", array(
            array(
                "methods" => WP_REST_Server::READABLE,
                "callback" => array($this, "get_css_variables_schema"),
                "permission_callback" => "__return_true" // Public endpoint
            ),
            "schema" => array($this, "get_schema_endpoint_schema")
            )
        );
    }
    
    /**
     * Get style kits
     */
    public function get_style_kits($request)
    {
        $args = array(
            "post_type" => "style_kit",
            "post_status" => array("publish", "alternative"),
            "posts_per_page" => $request->get_param("per_page") ?: 10,
            "paged" => $request->get_param("page") ?: 1
        );
        
        $query = new WP_Query($args);
        $style_kits = array();
        
        foreach ($query->posts as $post) {
            $style_kits[] = $this->prepare_item_for_response($post, $request);
        }
        
        $response = rest_ensure_response($style_kits);
        $response->header("X-WP-Total", (int) $query->found_posts);
        $response->header("X-WP-TotalPages", (int) $query->max_num_pages);
        
        return $response;
    }
    
    /**
     * Get single style kit
     */
    public function get_style_kit($request)
    {
        $post = get_post((int) $request["id"]);
        
        if (empty($post) || $post->post_type !== "style_kit") {
            return new WP_Error("wvc_style_kit_invalid_id", __("Invalid style kit ID.", "wvc-theme"), array("status" => 404));
        }
        
        return $this->prepare_item_for_response($post, $request);
    }
    
    /**
     * Create style kit
     */
    public function create_style_kit($request)
    {
        $prepared_post = $this->prepare_item_for_database($request);
        
        if (is_wp_error($prepared_post)) {
            return $prepared_post;
        }
        
        // If creating as published, first set status to draft to avoid triggering transition hook too early
        $is_publishing = ($prepared_post["post_status"] === "publish");
        if ($is_publishing) {
            $prepared_post["post_status"] = "draft";
        }
        
        $post_id = wp_insert_post($prepared_post, true);
        
        if (is_wp_error($post_id)) {
            return $post_id;
        }
        
        // Save CSS variables
        if ($request->get_param("css_variables")) {
            $css_variables = $this->validate_and_sanitize_variables($request->get_param("css_variables"));
            if (is_wp_error($css_variables)) {
                wp_delete_post($post_id, true);
                return $css_variables;
            }
            update_post_meta($post_id, "_wvc_css_variables", json_encode($css_variables));
        }
        
        // Save version
        if ($request->get_param("version")) {
            update_post_meta($post_id, "_wvc_version", sanitize_text_field($request->get_param("version")));
        }
        
        // Now publish if needed (this will trigger the status logic)
        if ($is_publishing) {
            wp_update_post(array(
                "ID" => $post_id,
                "post_status" => "publish"
            ));
        }
        
        $post = get_post($post_id);
        $response = $this->prepare_item_for_response($post, $request);
        $response->set_status(201);
        
        return $response;
    }
    
    /**
     * Update style kit
     */
    public function update_style_kit($request)
    {
        $post = get_post((int) $request["id"]);
        
        if (empty($post) || $post->post_type !== "style_kit") {
            return new WP_Error("wvc_style_kit_invalid_id", __("Invalid style kit ID.", "wvc-theme"), array("status" => 404));
        }
        
        $prepared_post = $this->prepare_item_for_database($request);
        
        if (is_wp_error($prepared_post)) {
            return $prepared_post;
        }
        
        $prepared_post["ID"] = $post->ID;
        $post_id = wp_update_post($prepared_post, true);
        
        if (is_wp_error($post_id)) {
            return $post_id;
        }
        
        // Update CSS variables
        if ($request->get_param("css_variables") !== null) {
            $css_variables = $this->validate_and_sanitize_variables($request->get_param("css_variables"));
            if (is_wp_error($css_variables)) {
                return $css_variables;
            }
            update_post_meta($post_id, "_wvc_css_variables", json_encode($css_variables));
        }
        
        // Update version
        if ($request->get_param("version")) {
            update_post_meta($post_id, "_wvc_version", sanitize_text_field($request->get_param("version")));
        }
        
        $post = get_post($post_id);
        return $this->prepare_item_for_response($post, $request);
    }
    
    /**
     * Delete style kit (hard delete only)
     */
    public function delete_style_kit($request)
    {
        $post = get_post((int) $request["id"]);
        
        if (empty($post) || $post->post_type !== "style_kit") {
            return new WP_Error("wvc_style_kit_invalid_id", __("Invalid style kit ID.", "wvc-theme"), array("status" => 404));
        }
        
        // Always hard delete style kits (no trash)
        $result = wp_delete_post($post->ID, true);
        
        if (!$result) {
            return new WP_Error("wvc_cannot_delete", __("The style kit cannot be deleted.", "wvc-theme"), array("status" => 500));
        }
        
        // Clear CSS variables cache
        delete_transient("wvc_css_variables_output");
        
        $response = $this->prepare_item_for_response($post, $request);
        $response->data["deleted"] = true;
        $response->data["previous"] = $response->data;
        
        return $response;
    }
    
    /**
     * Get CSS Variables Schema
     * Static endpoint for AI agents to understand the schema structure
     */
    public function get_css_variables_schema($request)
    {
        // Load the schema
        $schema = WVC_Style_Kit_Schema::load_schema();
        if (is_wp_error($schema)) {
            return $schema;
        }
        
        $response_data = array(
            "schema_info" => array(
                "title" => "CSS Variables Schema for Style Kits",
                "version" => "1.0.0",
                "description" => "Complete schema for creating and validating CSS variables in the WVC Theme Style Kit system",
                "endpoint" => rest_url($this->namespace . "/" . $this->rest_base . "/schema"),
                "usage" => "Use this schema to understand the structure and validation rules for CSS variables when creating style kits"
            ),
            "variable_structure" => array(
                "required_fields" => array("key", "value", "type"),
                "optional_fields" => array("category", "description", "default_value"),
                "field_descriptions" => array(
                    "key" => "Variable name without the -- prefix (e.g., 'primary-color'). Must be a valid CSS identifier.",
                    "value" => "The CSS value for the variable. Must be valid for the specified type.",
                    "type" => "The type of CSS value. Must be one of the allowed types.",
                    "category" => "Organizational category for grouping related variables. Must be one of the allowed categories.",
                    "description" => "Human-readable description of the variable's purpose and usage.",
                    "default_value" => "Fallback value to use if this variable is not set or is invalid."
                ),
                "validation_rules" => array(
                    "key" => "Must be unique within the style kit, valid CSS identifier format",
                    "value" => "Must match the specified type format and pass type-specific validation",
                    "type" => "Must be one of the predefined allowed types",
                    "category" => "Must be one of the predefined allowed categories if provided"
                )
            ),
            "allowed_types" => WVC_Style_Kit_Schema::get_allowed_types(),
            "allowed_categories" => WVC_Style_Kit_Schema::get_allowed_categories(),
            "type_definitions" => array(
                "color" => "CSS color values (hex, rgb, rgba, hsl, hsla, named colors)",
                "font-family" => "CSS font family declarations",
                "size" => "CSS length units (px, em, rem, %, vh, vw, etc.)",
                "number" => "Numeric values (unitless numbers)",
                "string" => "Text values (keywords, generic string values)",
                "url" => "CSS url() values for images and resources",
                "gradient" => "CSS gradient functions (linear-gradient, radial-gradient, etc.)",
                "time" => "Time values (s, ms)",
                "percentage" => "Percentage values (%)",
                "angle" => "Angle values (deg, rad, grad, turn)",
                "timing-function" => "CSS timing functions (ease, linear, cubic-bezier, etc.)",
                "shadow" => "CSS box-shadow and text-shadow values",
                "transform" => "CSS transform function values",
                "filter" => "CSS filter function values",
                "border" => "CSS border shorthand values",
                "keyword" => "CSS keyword values (auto, inherit, initial, etc.)"
            ),
            "type_examples" => array(
                "color" => array("#3b82f6", "rgb(59, 130, 246)", "rgba(59, 130, 246, 0.8)", "blue", "transparent"),
                "font-family" => array("'Inter', sans-serif", "Arial, Helvetica, sans-serif", "'Times New Roman', serif"),
                "size" => array("16px", "1.5rem", "100%", "50vh", "2em", "0"),
                "number" => array("1", "0.5", "10", "1.2", "0"),
                "string" => array("normal", "bold", "italic", "ease-in-out", "center"),
                "url" => array("url('/path/to/image.png')", "url('https://example.com/image.jpg')"),
                "gradient" => array("linear-gradient(45deg, #ff0000, #0000ff)", "radial-gradient(circle, #fff, #000)"),
                "time" => array("0.3s", "500ms", "1.5s", "0s"),
                "percentage" => array("50%", "100%", "25%", "0%"),
                "angle" => array("45deg", "0.5turn", "1.57rad", "90deg")
            ),
            "example_variables" => array(
                array(
                    "key" => "primary-color",
                    "value" => "#3b82f6",
                    "type" => "color",
                    "category" => "colors",
                    "description" => "Primary brand color used for buttons and links",
                    "default_value" => "#3b82f6"
                ),
                array(
                    "key" => "font-size-large",
                    "value" => "1.5rem",
                    "type" => "size",
                    "category" => "typography",
                    "description" => "Large font size for headings",
                    "default_value" => "1.5rem"
                ),
                array(
                    "key" => "transition-duration",
                    "value" => "0.3s",
                    "type" => "time",
                    "category" => "animation",
                    "description" => "Default transition duration for hover effects",
                    "default_value" => "0.3s"
                )
            ),
            "css_variables" => WVC_Style_Kit_Schema::get_css_variables(),
            "default_variables" => WVC_Style_Kit_Schema::get_default_variables()
        );
        
        return rest_ensure_response($response_data);
    }

    /**
     * Prepare item for response
     */
    protected function prepare_item_for_response($post, $request)
    {
        $css_variables_json = get_post_meta($post->ID, "_wvc_css_variables", true);
        $css_variables = $css_variables_json ? json_decode($css_variables_json, true) : array();
        
        $data = array(
            "id" => $post->ID,
            "title" => array(
                "rendered" => get_the_title($post),
                "raw" => $post->post_title
            ),
            "status" => $post->post_status,
            "date" => mysql_to_rfc3339($post->post_date),
            "date_gmt" => mysql_to_rfc3339($post->post_date_gmt),
            "modified" => mysql_to_rfc3339($post->post_modified),
            "modified_gmt" => mysql_to_rfc3339($post->post_modified_gmt),
            "css_variables" => $css_variables,
            "version" => get_post_meta($post->ID, "_wvc_version", true) ?: "1.0.0",
            "categories" => $this->get_variable_categories($css_variables)
        );
        
        $context = !empty($request["context"]) ? $request["context"] : "view";
        $data = $this->filter_response_by_context($data, $context);
        
        return rest_ensure_response($data);
    }
    
    /**
     * Prepare item for database
     */
    protected function prepare_item_for_database($request)
    {
        $prepared_post = array(
            "post_type" => "style_kit",
            "post_status" => "alternative"  // Default to alternative instead of publish
        );
        
        if ($request->get_param("title")) {
            $prepared_post["post_title"] = sanitize_text_field($request->get_param("title"));
        }
        
        if ($request->get_param("status")) {
            $status = sanitize_text_field($request->get_param("status"));
            // Only allow valid statuses: publish or alternative
            if (in_array($status, array("publish", "alternative"))) {
                $prepared_post["post_status"] = $status;
            }
        }
        
        return $prepared_post;
    }
    
    /**
     * Validate and sanitize variables
     */
    protected function validate_and_sanitize_variables($variables)
    {
        return WVC_Style_Kit_Schema::validate_and_sanitize_variables($variables);
    }
    
    /**
     * Get variable categories
     */
    protected function get_variable_categories($css_variables)
    {
        return WVC_Style_Kit_Schema::get_variable_categories($css_variables);
    }
    
    /**
     * Permission checks
     */
    public function get_items_permissions_check($request)
    {
        return current_user_can("read");
    }
    
    public function get_item_permissions_check($request)
    {
        return current_user_can("read");
    }
    
    public function create_item_permissions_check($request)
    {
        return current_user_can("edit_posts");
    }
    
    public function update_item_permissions_check($request)
    {
        if (!current_user_can("edit_posts")) {
            return false;
        }
        
        // Only allow editing of published style kits
        $post = get_post((int) $request["id"]);
        if (!$post || $post->post_type !== "style_kit") {
            return new WP_Error("wvc_style_kit_invalid_id", __("Invalid style kit ID.", "wvc-theme"), array("status" => 404));
        }
        
        if ($post->post_status !== "publish") {
            return new WP_Error("wvc_edit_restricted", __("Only active (published) style kits can be edited.", "wvc-theme"), array("status" => 403));
        }
        
        return true;
    }
    
    public function delete_item_permissions_check($request)
    {
        return current_user_can("delete_posts");
    }
    
    /**
     * Get collection parameters
     */
    public function get_collection_params()
    {
        return array(
            "page" => array(
                "description" => __("Current page of the collection.", "wvc-theme"),
                "type" => "integer",
                "default" => 1,
                "sanitize_callback" => "absint"
            ),
            "per_page" => array(
                "description" => __("Maximum number of items to be returned in result set.", "wvc-theme"),
                "type" => "integer",
                "default" => 10,
                "sanitize_callback" => "absint"
            )
        );
    }
    
    /**
     * Get endpoint args for item schema
     */
    public function get_endpoint_args_for_item_schema($method = WP_REST_Server::CREATABLE)
    {
        $args = array(
            "title" => array(
                "description" => __("The title for the style kit.", "wvc-theme"),
                "type" => "string",
                "sanitize_callback" => "sanitize_text_field"
            ),
            "status" => array(
                "description" => __("A named status for the style kit.", "wvc-theme"),
                "type" => "string",
                "enum" => array("publish", "alternative"),
                "sanitize_callback" => "sanitize_text_field"
            ),
            "css_variables" => array(
                "description" => __("CSS variables configuration.", "wvc-theme"),
                "type" => "array",
                "context" => array("view", "edit"),
                "items" => array(
                    "type" => "object",
                    "properties" => array(
                        "key" => array(
                            "type" => "string",
                            "description" => __("Variable key (without -- prefix)", "wvc-theme")
                        ),
                        "value" => array(
                            "type" => "string",
                            "description" => __("Variable value", "wvc-theme")
                        ),
                        "type" => array(
                            "type" => "string",
                            "enum" => WVC_Style_Kit_Schema::get_allowed_types(),
                            "description" => __("Variable type", "wvc-theme")
                        ),
                        "category" => array(
                            "type" => "string",
                            "enum" => WVC_Style_Kit_Schema::get_allowed_categories(),
                            "description" => __("Variable category", "wvc-theme")
                        ),
                        "description" => array(
                            "type" => "string",
                            "description" => __("Variable description", "wvc-theme")
                        ),
                        "default_value" => array(
                            "type" => "string",
                            "description" => __("Default value for the variable", "wvc-theme")
                        )
                    ),
                    "required" => array("key", "value", "type")
                )
            ),
            "version" => array(
                "description" => __("Style kit version.", "wvc-theme"),
                "type" => "string",
                "sanitize_callback" => "sanitize_text_field"
            )
        );
        
        if ($method === WP_REST_Server::CREATABLE) {
            $args["title"]["required"] = true;
        }
        
        return $args;
    }
    
    /**
     * Filter response by context
     */
    protected function filter_response_by_context($data, $context)
    {
        if ($context === "edit") {
            // Return all fields for edit context
            return $data;
        }
        
        // For view context, remove raw content-managers
        if (isset($data["title"]["raw"])) {
            unset($data["title"]["raw"]);
        }
        
        return $data;
    }
    
    /**
     * Get context parameter
     */
    protected function get_context_param($args = array())
    {
        $param_details = array(
            "description" => __("Scope under which the request is made; determines fields present in response.", "wvc-theme"),
            "type" => "string",
            "sanitize_callback" => "sanitize_key"
        );
        
        $param_details["enum"] = array("view", "edit");
        
        if (isset($args["default"])) {
            $param_details["default"] = $args["default"];
        }
        
        return $param_details;
    }
    
    /**
     * Get public item schema
     */
    public function get_public_item_schema()
    {
        $schema = array(
            '$schema' => "http://json-schema.org/draft-04/schema#",
            "title" => "style_kit",
            "type" => "object",
            "properties" => array(
                "id" => array(
                    "description" => __("Unique identifier for the style kit.", "wvc-theme"),
                    "type" => "integer",
                    "context" => array("view", "edit"),
                    "readonly" => true
                ),
                "title" => array(
                    "description" => __("The title for the style kit.", "wvc-theme"),
                    "type" => "object",
                    "context" => array("view", "edit"),
                    "properties" => array(
                        "raw" => array(
                            "description" => __("Title for the style kit, as it exists in the database.", "wvc-theme"),
                            "type" => "string",
                            "context" => array("edit")
                        ),
                        "rendered" => array(
                            "description" => __("HTML title for the style kit, transformed for display.", "wvc-theme"),
                            "type" => "string",
                            "context" => array("view", "edit"),
                            "readonly" => true
                        )
                    )
                ),
                "status" => array(
                    "description" => __("Status of the style kit.", "wvc-theme"),
                    "type" => "string",
                    "enum" => array("publish", "draft", "private"),
                    "context" => array("view", "edit")
                ),
                "date" => array(
                    "description" => __("The date the style kit was published, in the site's timezone.", "wvc-theme"),
                    "type" => "string",
                    "format" => "date-time",
                    "context" => array("view", "edit"),
                    "readonly" => true
                ),
                "date_gmt" => array(
                    "description" => __("The date the style kit was published, as GMT.", "wvc-theme"),
                    "type" => "string",
                    "format" => "date-time",
                    "context" => array("view", "edit"),
                    "readonly" => true
                ),
                "modified" => array(
                    "description" => __("The date the style kit was last modified, in the site's timezone.", "wvc-theme"),
                    "type" => "string",
                    "format" => "date-time",
                    "context" => array("view", "edit"),
                    "readonly" => true
                ),
                "modified_gmt" => array(
                    "description" => __("The date the style kit was last modified, as GMT.", "wvc-theme"),
                    "type" => "string",
                    "format" => "date-time",
                    "context" => array("view", "edit"),
                    "readonly" => true
                ),
                "css_variables" => array(
                    "description" => __("CSS variables configuration.", "wvc-theme"),
                    "type" => "array",
                    "context" => array("view", "edit"),
                    "items" => array(
                        "type" => "object",
                        "properties" => array(
                            "key" => array(
                                "type" => "string",
                                "description" => __("Variable key (without -- prefix)", "wvc-theme")
                            ),
                            "value" => array(
                                "type" => "string",
                                "description" => __("Variable value", "wvc-theme")
                            ),
                            "type" => array(
                                "type" => "string",
                                "enum" => WVC_Style_Kit_Schema::get_allowed_types(),
                                "description" => __("Variable type", "wvc-theme")
                            ),
                            "category" => array(
                                "type" => "string",
                                "enum" => WVC_Style_Kit_Schema::get_allowed_categories(),
                                "description" => __("Variable category", "wvc-theme")
                            ),
                            "description" => array(
                                "type" => "string",
                                "description" => __("Variable description", "wvc-theme")
                            ),
                            "default_value" => array(
                                "type" => "string",
                                "description" => __("Default value for the variable", "wvc-theme")
                            )
                        ),
                        "required" => array("key", "value", "type")
                    )
                ),
                "version" => array(
                    "description" => __("Style kit version.", "wvc-theme"),
                    "type" => "string",
                    "context" => array("view", "edit")
                ),
                "categories" => array(
                    "description" => __("Categories of CSS variables in the style kit.", "wvc-theme"),
                    "type" => "array",
                    "context" => array("view", "edit"),
                    "readonly" => true
                )
            )
        );
        
        return $schema;
    }

    /**
     * Get schema for the schema endpoint itself
     */
    public function get_schema_endpoint_schema()
    {
        $schema = array(
            '$schema' => "http://json-schema.org/draft-04/schema#",
            "title" => "css_variables_schema",
            "type" => "object",
            "description" => "Schema information for CSS variables in Style Kits",
            "properties" => array(
                "schema_info" => array(
                    "description" => __("Information about the schema itself", "wvc-theme"),
                    "type" => "object",
                    "properties" => array(
                        "title" => array("type" => "string"),
                        "version" => array("type" => "string"),
                        "description" => array("type" => "string"),
                        "endpoint" => array("type" => "string"),
                        "usage" => array("type" => "string")
                    )
                ),
                "variable_structure" => array(
                    "description" => __("Structure and validation rules for CSS variables", "wvc-theme"),
                    "type" => "object",
                    "properties" => array(
                        "required_fields" => array(
                            "type" => "array",
                            "items" => array("type" => "string")
                        ),
                        "optional_fields" => array(
                            "type" => "array",
                            "items" => array("type" => "string")
                        ),
                        "field_descriptions" => array(
                            "type" => "object"
                        ),
                        "validation_rules" => array(
                            "type" => "object"
                        )
                    )
                ),
                "allowed_types" => array(
                    "description" => __("List of allowed CSS variable types", "wvc-theme"),
                    "type" => "array",
                    "items" => array("type" => "string")
                ),
                "allowed_categories" => array(
                    "description" => __("List of allowed variable categories", "wvc-theme"),
                    "type" => "array",
                    "items" => array("type" => "string")
                ),
                "type_definitions" => array(
                    "description" => __("Definitions for each allowed type", "wvc-theme"),
                    "type" => "object"
                ),
                "type_examples" => array(
                    "description" => __("Examples for each type", "wvc-theme"),
                    "type" => "object"
                ),
                "example_variables" => array(
                    "description" => __("Complete example variables", "wvc-theme"),
                    "type" => "array",
                    "items" => array(
                        "type" => "object",
                        "properties" => array(
                            "key" => array("type" => "string"),
                            "value" => array("type" => "string"),
                            "type" => array("type" => "string"),
                            "category" => array("type" => "string"),
                            "description" => array("type" => "string"),
                            "default_value" => array("type" => "string")
                        )
                    )
                ),
                "css_variables" => array(
                    "description" => __("All CSS variables from the schema file", "wvc-theme"),
                    "type" => "array"
                ),
                "default_variables" => array(
                    "description" => __("Default variables with their default values", "wvc-theme"),
                    "type" => "array"
                )
            )
        );

        return $schema;
    }
}

// Initialize the REST API
new WVC_Style_Kit_REST_API(); 