<?php
/**
 * Style Kit Custom Post Type
 *
 * This class handles registration and management of the Style Kit custom post type.
 * It manages post status, meta fields, admin interface, and ensures only one
 * published style kit exists at a time. Includes REST API support and CSS
 * variable management functionality.
 *
 * @package    WVC_Theme
 * @subpackage Style_Kits
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 * @link       https://github.com/your-repo/wvc-theme
 */

if (!defined("ABSPATH")) {
    exit;
}

class WVC_Style_Kit_Post_Type
{
    
    public function __construct()
    {
        add_action("init", array($this, "register_post_status"));
        add_action("init", array($this, "register_post_type"));
        add_action("init", array($this, "register_meta_fields"));
        add_action("add_meta_boxes", array($this, "add_meta_boxes"));
        add_action("save_post", array($this, "save_meta"));
        add_action("transition_post_status", array($this, "handle_status_transition"), 10, 3);
        
        add_filter("manage_style_kit_posts_columns", array($this, "add_status_column"));
        add_action("manage_style_kit_posts_custom_column", array($this, "show_status_column"), 10, 2);
        add_action("admin_head", array($this, "add_admin_styles"));
        add_action("current_screen", array($this, "restrict_editing_to_published"));
    }
    
    /**
     * Register custom post status for Style Kits
     */
    public function register_post_status()
    {
        register_post_status("alternative", array(
            "label" => __("Alternative", "wvc-theme"),
            "public" => false,
            "show_in_admin_all_list" => true,
            "show_in_admin_status_list" => true,
            "exclude_from_search" => true,
            "label_count" => _n_noop(
                "Alternative <span class=\"count\">(%s)</span>",
                "Alternative <span class=\"count\">(%s)</span>",
                "wvc-theme"
            ),
        ));
    }
    
    /**
     * Handle status transitions to ensure only one published style kit
     */
    public function handle_status_transition($new_status, $old_status, $post)
    {
        // Only handle style_kit post type
        if ($post->post_type !== "style_kit") {
            return;
        }
        
        // If a style kit is being published, set all others to alternative
        if ($new_status === "publish" && $old_status !== "publish") {
            $this->set_other_kits_to_alternative($post->ID);
        }
    }
    
    /**
     * Set all other style kits to alternative status
     */
    private function set_other_kits_to_alternative($current_kit_id)
    {
        global $wpdb;
        
        // Update all other published style kits to alternative status
        $wpdb->update(
            $wpdb->posts,
            array("post_status" => "alternative"),
            array(
                "post_type" => "style_kit",
                "post_status" => "publish"
            ),
            array("%s"),
            array("%s", "%s")
        );
        
        // Then set the current kit back to published
        $wpdb->update(
            $wpdb->posts,
            array("post_status" => "publish"),
            array("ID" => $current_kit_id),
            array("%s"),
            array("%d")
        );
        
        // Clear CSS variables cache
        delete_transient("wvc_css_variables_output");
    }
    
    /**
     * Register Style Kit Custom Post Type
     */
    public function register_post_type()
    {
        $labels = array(
            "name" => __("Style Kits", "wvc-theme"),
            "singular_name" => __("Style Kit", "wvc-theme"),
            "menu_name" => __("Style Kits", "wvc-theme"),
            "add_new" => __("Add New", "wvc-theme"),
            "add_new_item" => __("Add New Style Kit", "wvc-theme"),
            "new_item" => __("New Style Kit", "wvc-theme"),
            "edit_item" => __("Edit Style Kit", "wvc-theme"),
            "view_item" => __("View Style Kit", "wvc-theme"),
            "all_items" => __("All Style Kits", "wvc-theme"),
            "search_items" => __("Search Style Kits", "wvc-theme"),
            "parent_item_colon" => __("Parent Style Kit:", "wvc-theme"),
            "not_found" => __("No style kits found.", "wvc-theme"),
            "not_found_in_trash" => __("No style kits found in Trash.", "wvc-theme"),
            "featured_image" => __("Style Kit Preview", "wvc-theme"),
            "set_featured_image" => __("Set preview image", "wvc-theme"),
            "remove_featured_image" => __("Remove preview image", "wvc-theme"),
            "use_featured_image" => __("Use as preview image", "wvc-theme"),
            "archives" => __("Style Kit Archives", "wvc-theme"),
            "insert_into_item" => __("Insert into style kit", "wvc-theme"),
            "uploaded_to_this_item" => __("Uploaded to this style kit", "wvc-theme"),
            "filter_items_list" => __("Filter style kits list", "wvc-theme"),
            "items_list_navigation" => __("Style kits list navigation", "wvc-theme"),
            "items_list" => __("Style kits list", "wvc-theme"),
        );

        $args = array(
            "labels" => $labels,
            "public" => false,
            "publicly_queryable" => false,
            "show_ui" => true,
            "show_in_menu" => true,
            "query_var" => false,
            "rewrite" => array("slug" => "style-kit"),
            "capability_type" => "post",
            "capabilities" => array(
                "create_posts" => "do_not_allow", // Disable add new - only via REST
            ),
            "map_meta_cap" => true,
            "has_archive" => false,
            "hierarchical" => false,
            "menu_position" => 25,
            "menu_icon" => "dashicons-art",
            "supports" => array("title"),
            "show_in_rest" => true,
            "rest_base" => "style-kits",
            "rest_controller_class" => "WP_REST_Posts_Controller",
        );

        register_post_type("style_kit", $args);
    }
    
    /**
     * Register meta fields for style kit
     */
    public function register_meta_fields()
    {
        register_post_meta(
            "style_kit", "_wvc_css_content", array(
            "type" => "string",
            "description" => "Complete CSS stylesheet content",
            "single" => true,
            "show_in_rest" => true,
            "sanitize_callback" => array($this, "sanitize_css_content"),
            "auth_callback" => function () {
                return current_user_can("edit_posts");
            }
            )
        );
        
        register_post_meta(
            "style_kit", "_wvc_version", array(
            "type" => "string",
            "description" => "Style kit version",
            "single" => true,
            "show_in_rest" => true,
            "default" => "1.0.0",
            "sanitize_callback" => "sanitize_text_field"
            )
        );
    }
    
    /**
     * Add meta boxes for style kit admin
     */
    public function add_meta_boxes()
    {
        add_meta_box(
            "wvc_css_content",
            __("CSS Content", "wvc-theme"),
            array($this, "css_content_meta_box_callback"),
            "style_kit",
            "normal",
            "high"
        );
    }
    
    /**
     * Meta box callback for CSS content
     */
    public function css_content_meta_box_callback($post)
    {
        wp_nonce_field("wvc_save_style_kit_meta", "wvc_style_kit_nonce");
        
        $css_content = get_post_meta($post->ID, "_wvc_css_content", true);
        
        echo "<div id=\"wvc-css-content-editor\">";
        echo "<p><strong>" . __("CSS Stylesheet Content", "wvc-theme") . "</strong></p>";
        echo "<textarea id=\"wvc_css_content\" name=\"wvc_css_content\" rows=\"20\" cols=\"100\" style=\"width: 100%; font-family: monospace;\">";
        echo esc_textarea($css_content);
        echo "</textarea>";
        echo "<p class=\"description\">" . __("Enter the complete CSS stylesheet content. This will be used as the full stylesheet for the style kit.", "wvc-theme") . "</p>";
        echo "</div>";
        
        // Add JavaScript for basic syntax highlighting
        ?>
        <script>
        document.addEventListener("DOMContentLoaded", function() {
            const textarea = document.getElementById("wvc_css_content");
            if (textarea) {
                // Basic CSS syntax highlighting could be added here
                textarea.style.borderColor = "#ddd";
            }
        });
        </script>
        <style>
        #wvc-css-content-editor textarea {
            border: 1px solid #ddd;
            border-radius: 3px;
            padding: 10px;
        }
        </style>
        <?php
    }
    
    /**
     * Save style kit meta
     */
    public function save_meta($post_id)
    {
        if (!isset($_POST["wvc_style_kit_nonce"]) || !wp_verify_nonce($_POST["wvc_style_kit_nonce"], "wvc_save_style_kit_meta")) {
            return;
        }
        
        if (defined("DOING_AUTOSAVE") && DOING_AUTOSAVE) {
            return;
        }
        
        if (!current_user_can("edit_post", $post_id)) {
            return;
        }
        
        if (isset($_POST["wvc_css_content"])) {
            $css_content = stripslashes($_POST["wvc_css_content"]);
            update_post_meta($post_id, "_wvc_css_content", $css_content);
        }
    }
    
    /**
     * Sanitize CSS content
     */
    public function sanitize_css_content($value)
    {
        if (empty($value)) {
            return "";
        }
        
        // Basic CSS content sanitization - allow CSS syntax
        return wp_kses($value, array());
    }
    
    /**
     * Add alternative status to post status dropdown in admin
     */
    public function add_status_to_dropdown()
    {
        global $post;
        
        // Only add for style_kit post type
        if (!$post || $post->post_type !== "style_kit") {
            return;
        }
        
        ?>
        <script>
        jQuery(document).ready(function($) {
            // Only show publish/alternative statuses
            if ($('#post_status').length) {
                // Remove other status options
                $('#post_status option').each(function() {
                    var val = $(this).val();
                    if (val !== 'publish' && val !== 'alternative' && val !== 'draft') {
                        $(this).remove();
                    }
                });
                
                // Add alternative status if not present
                if ($('#post_status option[value="alternative"]').length === 0) {
                    $('#post_status').append('<option value="alternative">Alternative</option>');
                }
                
                // If current post is alternative, select it
                if ('<?php echo $post->post_status; ?>' === 'alternative') {
                    $('#post_status').val('alternative');
                    $('.misc-pub-section label').text('Status: Alternative');
                }
            }
            
            // Update status text when changed
            $('#post_status').on('change', function() {
                if ($(this).val() === 'alternative') {
                    $('.misc-pub-section label').text('Status: Alternative');
                } else if ($(this).val() === 'publish') {
                    $('.misc-pub-section label').text('Status: Published');
                }
            });
        });
        </script>
        <?php
    }
    
    /**
     * Add status column to admin list
     */
    public function add_status_column($columns)
    {
        $columns["wvc_status"] = __("Status", "wvc-theme");
        return $columns;
    }
    
    /**
     * Show status column content-managers
     */
    public function show_status_column($column, $post_id)
    {
        if ($column === "wvc_status") {
            $post = get_post($post_id);
            $status = $post->post_status;
            
            if ($status === "publish") {
                echo '<span class="wvc-status-active">● Active</span>';
            } else {
                echo '<span class="wvc-status-alternative">○ Alternative</span>';
            }
        }
    }
    
    /**
     * Add admin styles for visual indicators
     */
    public function add_admin_styles()
    {
        global $current_screen;
        
        if (!$current_screen || $current_screen->post_type !== "style_kit") {
            return;
        }
        
        ?>
        <style>
        /* Style Kit status indicators */
        .wvc-status-active {
            color: #46b450;
            font-weight: 600;
        }
        
        .wvc-status-alternative {
            color: #999;
        }
        
        /* Highlight active style kit row */
        .post-type-style_kit .wp-list-table tbody tr[data-status="publish"] {
            background-color: #f0f8ff;
            border-left: 4px solid #46b450;
        }
        
        /* Alternative kit styling */
        .post-type-style_kit .wp-list-table tbody tr[data-status="alternative"] {
            opacity: 0.7;
        }
        
        /* Admin edit restriction notice */
        .wvc-edit-restriction {
            background: #fff3cd;
            border: 1px solid #ffeaa7;
            border-radius: 4px;
            padding: 12px;
            margin: 20px 0;
            color: #856404;
        }
        </style>
        
        <script>
        jQuery(document).ready(function($) {
            // Add data attributes for styling
            $('.wp-list-table tbody tr').each(function() {
                var statusText = $(this).find('.wvc-status-active, .wvc-status-alternative').text();
                if (statusText.includes('Active')) {
                    $(this).attr('data-status', 'publish');
                } else if (statusText.includes('Alternative')) {
                    $(this).attr('data-status', 'alternative');
                }
            });
        });
        </script>
        <?php
    }
    
    /**
     * Restrict editing to only published style kits
     */
    public function restrict_editing_to_published()
    {
        global $current_screen, $post;
        
        if (!$current_screen || $current_screen->id !== "style_kit") {
            return;
        }
        
        // Only restrict on edit screen, not on list screen
        if ($current_screen->action !== "add" && isset($_GET["post"])) {
            $post_id = intval($_GET["post"]);
            $style_kit = get_post($post_id);
            
            if ($style_kit && $style_kit->post_status !== "publish") {
                add_action("admin_notices", array($this, "show_edit_restriction_notice"));
                add_action("admin_head", array($this, "disable_editing_form"));
            }
        }
    }
    
    /**
     * Show notice about editing restriction
     */
    public function show_edit_restriction_notice()
    {
        ?>
        <div class="notice notice-warning wvc-edit-restriction">
            <p><strong><?php _e("Editing Restricted", "wvc-theme"); ?></strong></p>
            <p><?php _e("Only the active (published) style kit can be edited. This style kit is currently alternative. To edit this kit, first publish it via the REST API.", "wvc-theme"); ?></p>
        </div>
        <?php
    }
    
    /**
     * Disable form elements for non-published kits
     */
    public function disable_editing_form()
    {
        ?>
        <script>
        jQuery(document).ready(function($) {
            // Disable all form inputs except status
            $('#post input, #post textarea, #post select').not('#post_status').prop('disabled', true);
            
            // Disable meta boxes
            $('.postbox input, .postbox textarea, .postbox select').prop('disabled', true);
            
            // Hide publish/update button
            $('#publish, #save-post').hide();
            
            // Show explanation
            $('#publishing-action').append('<p style="color: #999; font-style: italic;">Publishing disabled - kit not active</p>');
        });
        </script>
        <?php
    }

} 