<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * WVC SVG Upload Handler
 *
 * Handles SVG uploads by allowing SVG mime types and sanitizing SVG files
 * before they are uploaded to the media library.
 *
 * @package WVC_Theme
 * @subpackage Import
 * @since 1.0.0
 */
class WVC_SVG_Upload_Handler {

    /**
     * @var WVC_SVG_Upload_Handler
     */
    protected static $instance = null;

    /**
     * Get singleton instance
     *
     * @return WVC_SVG_Upload_Handler
     */
    public static function getInstance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }

        return self::$instance;
    }

    /**
     * Constructor
     */
    private function __construct() {
        // Allow SVG uploads
        add_filter('upload_mimes', [$this, 'allowSvgUpload']);
        
        // Fix uploads for specific WordPress versions
        add_filter('wp_check_filetype_and_ext', [$this, 'allowSvgUploadWpFix'], 10, 4);
        
        // Sanitize SVG files before upload
        add_filter('wp_handle_upload_prefilter', [$this, 'sanitizeSvg']);
    }

    /**
     * Fixes uploads for 4.7.1 and 4.7.2 wp versions
     *
     * @param array  $data     File data
     * @param string $file     File path
     * @param string $filename File name
     * @param array  $mimes    Allowed mime types
     * @return array
     */
    public function allowSvgUploadWpFix($data, $file, $filename, $mimes) {
        if ($this->visibilityCheck()) {
            global $wp_version;
            if ($wp_version !== '4.7.1' || $wp_version !== '4.7.2') {
                return $data;
            }

            $filetype = wp_check_filetype($filename, $mimes);

            return [
                'ext'             => $filetype['ext'],
                'type'            => $filetype['type'],
                'proper_filename' => $data['proper_filename']
            ];
        }

        return $data;
    }

    /**
     * Allows to upload svg files
     *
     * @param array $mimes Allowed mime types
     * @return array
     */
    public function allowSvgUpload($mimes) {
        if ($this->visibilityCheck()) {
            $mimes['svg'] = 'image/svg+xml';
        }
        return $mimes;
    }

    /**
     * Sanitize SVG file before upload
     *
     * @param array $file File data
     * @return array
     */
    public function sanitizeSvg($file) {
        if ($file['type'] !== 'image/svg+xml') {
            return $file;
        }

        // Check if our SVG sanitizer class exists
        if (!class_exists('WVC_SVG_Sanitizer')) {
            $file['error'] = "Sorry, SVG sanitizer is not found";
            return $file;
        }

        // Read the file content
        $file_content = file_get_contents($file['tmp_name']);
        if (false === $file_content) {
            $file['error'] = "Sorry, couldn't read the SVG file";
            return $file;
        }

        // Sanitize the SVG content
        $sanitizer = new WVC_SVG_Sanitizer();
        $sanitized_content = $sanitizer->sanitize_file($file_content);
        
        if (null === $sanitized_content) {
            $file['error'] = "Sorry, this file couldn't be sanitized and wasn't uploaded";
            return $file;
        }

        // Write the sanitized content back to the file
        $result = file_put_contents($file['tmp_name'], $sanitized_content);
        if (false === $result) {
            $file['error'] = "Sorry, couldn't write sanitized SVG content";
            return $file;
        }

        return $file;
    }

    /**
     * Check if SVG uploads should be allowed
     *
     * @return bool
     */
    private function visibilityCheck() {
        return apply_filters('wvc_svg_support', current_user_can('unfiltered_html'));
    }

    /**
     * Enable SVG support for the current request
     */
    public function enableSvgSupport() {
        // This method can be called to ensure SVG support is enabled
        // even if the class wasn't instantiated through the normal WordPress hooks
    }
}
