<?php

namespace WVC\FormSubmissions;

/**
 * WVC Form Submissions Helper Functions
 *
 * Helper functions for querying and retrieving form submission data
 *
 * @package    WVC_Theme
 * @subpackage FormSubmissions
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.16
 */

// Prevent direct access
if ( ! defined("ABSPATH")) {
    exit;
}

/**
 * Helper function to get form submissions by form ID
 */
function wvc_get_form_submissions_by_form_id($form_id, $limit = 10) {
    $args = array(
        'post_type' => 'wvc_form_data',
        'post_status' => 'private',
        'posts_per_page' => $limit,
        'meta_query' => array(
            array(
                'key' => '_wvc_form_id',
                'value' => $form_id,
                'compare' => '='
            )
        ),
        'orderby' => 'date',
        'order' => 'DESC'
    );
    
    return get_posts($args);
}

/**
 * Helper function to get form submissions by section name
 */
function wvc_get_form_submissions_by_section($section_name, $limit = 10) {
    $args = array(
        'post_type' => 'wvc_form_data',
        'post_status' => 'private',
        'posts_per_page' => $limit,
        'meta_query' => array(
            array(
                'key' => '_wvc_section_name',
                'value' => $section_name,
                'compare' => '='
            )
        ),
        'orderby' => 'date',
        'order' => 'DESC'
    );
    
    return get_posts($args);
}

/**
 * Helper function to get form submissions by specific field value
 */
function wvc_get_form_submissions_by_field($field_name, $field_value, $limit = 10) {
    $args = array(
        'post_type' => 'wvc_form_data',
        'post_status' => 'private',
        'posts_per_page' => $limit,
        'meta_query' => array(
            array(
                'key' => '_wvc_field_' . sanitize_key($field_name),
                'value' => $field_value,
                'compare' => '='
            )
        ),
        'orderby' => 'date',
        'order' => 'DESC'
    );
    
    return get_posts($args);
}

/**
 * Helper function to search form submissions by multiple criteria
 */
function wvc_search_form_submissions($criteria = array(), $limit = 10) {
    $meta_query = array('relation' => 'AND');
    
    // Add form ID filter if provided
    if (!empty($criteria['form_id'])) {
        $meta_query[] = array(
            'key' => '_wvc_form_id',
            'value' => $criteria['form_id'],
            'compare' => '='
        );
    }
    
    // Add section filter if provided
    if (!empty($criteria['section_name'])) {
        $meta_query[] = array(
            'key' => '_wvc_section_name',
            'value' => $criteria['section_name'],
            'compare' => '='
        );
    }
    
    // Add field-specific filters
    if (!empty($criteria['fields']) && is_array($criteria['fields'])) {
        foreach ($criteria['fields'] as $field_name => $field_value) {
            $meta_query[] = array(
                'key' => '_wvc_field_' . sanitize_key($field_name),
                'value' => $field_value,
                'compare' => 'LIKE'
            );
        }
    }
    
    // Add date range filter if provided
    if (!empty($criteria['date_from']) || !empty($criteria['date_to'])) {
        $date_query = array();
        if (!empty($criteria['date_from'])) {
            $date_query['after'] = $criteria['date_from'];
        }
        if (!empty($criteria['date_to'])) {
            $date_query['before'] = $criteria['date_to'];
        }
    }
    
    $args = array(
        'post_type' => 'wvc_form_data',
        'post_status' => 'private',
        'posts_per_page' => $limit,
        'meta_query' => $meta_query,
        'orderby' => 'date',
        'order' => 'DESC'
    );
    
    // Add date query if specified
    if (!empty($date_query)) {
        $args['date_query'] = array($date_query);
    }
    
    return get_posts($args);
}

/**
 * Helper function to get form fields from the first submission of a specific form ID
 */
function wvc_get_form_fields_by_form_id($form_id) {
    // Get all submissions for this form ID to extract all field names
    $submissions = wvc_get_form_submissions_by_form_id($form_id, -1);
    
    if (empty($submissions)) {
        return array();
    }
    
    $all_fields = array();
    
    // Loop through all submissions to collect all unique field names
    foreach ($submissions as $submission) {
        $post_id = $submission->ID;
        $form_data_json = get_post_meta($post_id, '_wvc_form_data', true);
        $form_data = $form_data_json ? json_decode($form_data_json, true) : array();
        
        // If JSON data is available, merge the keys
        if (!empty($form_data)) {
            $all_fields = array_merge($all_fields, array_keys($form_data));
        } else {
            // Fallback: get from individual field meta
            $all_meta = get_post_meta($post_id);
            foreach ($all_meta as $meta_key => $meta_value) {
                if (strpos($meta_key, '_wvc_field_') === 0) {
                    $field_name = str_replace('_wvc_field_', '', $meta_key);
                    $all_fields[] = $field_name;
                }
            }
        }
    }
    
    // Return unique field names
    return array_unique($all_fields);
}

/**
 * Helper function to get all submissions for a form ID with their field data
 */
function wvc_get_form_submissions_with_fields($form_id, $limit = -1) {
    $submissions = wvc_get_form_submissions_by_form_id($form_id, $limit);
    $submissions_data = array();
    
    foreach ($submissions as $submission) {
        $submission_data = wvc_get_form_submission_data($submission->ID);
        $submissions_data[] = $submission_data;
    }
    
    return $submissions_data;
}

/**
 * Helper function to get distinct form IDs with submission counts
 */
function wvc_get_distinct_form_ids_with_counts() {
    global $wpdb;
    
    $query = "
        SELECT pm.meta_value as form_id, COUNT(*) as submission_count
        FROM {$wpdb->postmeta} pm
        INNER JOIN {$wpdb->posts} p ON pm.post_id = p.ID
        WHERE pm.meta_key = '_wvc_form_id'
        AND p.post_type = 'wvc_form_data'
        AND p.post_status = 'private'
        GROUP BY pm.meta_value
        ORDER BY submission_count DESC, pm.meta_value ASC
    ";
    
    $results = $wpdb->get_results($query);
    
    $form_data = array();
    foreach ($results as $result) {
        $form_data[] = array(
            'form_id' => $result->form_id,
            'submission_count' => intval($result->submission_count)
        );
    }
    
    return $form_data;
}

/**
 * Helper function to get submission count for a specific form ID
 */
function wvc_get_form_submission_count($form_id) {
    $args = array(
        'post_type' => 'wvc_form_data',
        'post_status' => 'private',
        'posts_per_page' => -1,
        'meta_query' => array(
            array(
                'key' => '_wvc_form_id',
                'value' => $form_id,
                'compare' => '='
            )
        ),
        'fields' => 'ids'
    );
    
    $posts = get_posts($args);
    return count($posts);
}

/**
 * Helper function to get total form submissions count
 */
function wvc_get_total_submissions_count() {
    $count = wp_count_posts('wvc_form_data');
    return $count->private ?? 0;
}

/**
 * Helper function to get form data from a submission post
 */
function wvc_get_form_submission_data($post_id) {
    $form_data_json = get_post_meta($post_id, '_wvc_form_data', true);
    $form_data = $form_data_json ? json_decode($form_data_json, true) : array();
    
    // Get the complete form data structure
    $form_data_complete = get_post_meta($post_id, '_wvc_form_data_complete', true);
    
    // Get submission metadata
    $submission_metadata = get_post_meta($post_id, '_wvc_submission_metadata', true);
    
    // Get individual form fields
    $individual_fields = array();
    $all_meta = get_post_meta($post_id);
    foreach ($all_meta as $meta_key => $meta_value) {
        if (strpos($meta_key, '_wvc_field_') === 0) {
            $field_name = str_replace('_wvc_field_', '', $meta_key);
            $individual_fields[$field_name] = $meta_value[0];
        }
    }
    
    return array(
        'form_id' => get_post_meta($post_id, '_wvc_form_id', true),
        'section_name' => get_post_meta($post_id, '_wvc_section_name', true),
        'timestamp' => get_post_meta($post_id, '_wvc_timestamp', true),
        'session_id' => get_post_meta($post_id, '_wvc_session_id', true),
        'page_url' => get_post_meta($post_id, '_wvc_page_url', true),
        'user_agent' => get_post_meta($post_id, '_wvc_user_agent', true),
        'submission_attempt' => get_post_meta($post_id, '_wvc_submission_attempt', true),
        'form_version' => get_post_meta($post_id, '_wvc_form_version', true),
        'validation_errors' => get_post_meta($post_id, '_wvc_validation_errors', true),
        'form_data' => $form_data, // Original JSON format for backward compatibility
        'form_data_complete' => $form_data_complete, // Array format
        'individual_fields' => $individual_fields, // Individual field values
        'submission_metadata' => $submission_metadata, // Additional metadata
        'submission_date' => get_the_date('Y-m-d H:i:s', $post_id)
    );
}