<?php
/**
 * Template Custom Post Type
 *
 * This class handles registration and management of the Template custom post type
 * for bulk operations. It manages template metadata, validation, and provides
 * REST API support for template creation and management.
 *
 * @package    WVC_Theme
 * @subpackage Bulk_Operations
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 * @link       https://github.com/your-repo/wvc-theme
 */

if (!defined("ABSPATH")) {
    exit;
}

class WVC_Template_Post_Type
{
    
    /**
     * WordPress template types with their hierarchy relationships
     */
    private $template_hierarchy = array(
        "index" => array("parent" => null, "description" => "Root fallback template for all pages"),
        "singular" => array("parent" => "index", "description" => "Parent template for single entries"),
        "single" => array("parent" => "singular", "description" => "Single posts template"),
        "page" => array("parent" => "singular", "description" => "Static pages template"),
        "attachment" => array("parent" => "singular", "description" => "Attachment pages template"),
        "archive" => array("parent" => "index", "description" => "Parent template for archives"),
        "category" => array("parent" => "archive", "description" => "Category archive template"),
        "tag" => array("parent" => "archive", "description" => "Tag archive template"),
        "author" => array("parent" => "archive", "description" => "Author archive template"),
        "date" => array("parent" => "archive", "description" => "Date archive template"),
        "taxonomy" => array("parent" => "archive", "description" => "Custom taxonomy archive template"),
        "search" => array("parent" => "index", "description" => "Search results template"),
        "404" => array("parent" => "index", "description" => "404 error page template"),
        "home" => array("parent" => "index", "description" => "Blog posts index template"),
        "front-page" => array("parent" => "home", "description" => "Front page template"),
        "privacy-policy" => array("parent" => "page", "description" => "Privacy policy page template"),
    );

    private $template_types = array(
        "header",
        "footer",
        "content",
        "sidebar"
    );
    
    public function __construct()
    {
        add_action("init", array($this, "register_post_type"));
        add_action("init", array($this, "register_meta_fields"));
        add_action("add_meta_boxes", array($this, "add_meta_boxes"));
        add_action("save_post", array($this, "save_meta"));
    }
    
    /**
     * Register Template Hierarchy Custom Post Type
     */
    public function register_post_type()
    {
        $labels = array(
            "name" => __("Templates", "wvc-theme"),
            "singular_name" => __("Template", "wvc-theme"),
            "menu_name" => __("Template Hierarchy", "wvc-theme"),
            "add_new" => __("Add New", "wvc-theme"),
            "add_new_item" => __("Add New Template", "wvc-theme"),
            "new_item" => __("New Template", "wvc-theme"),
            "edit_item" => __("Edit Template", "wvc-theme"),
            "view_item" => __("View Template", "wvc-theme"),
            "all_items" => __("All Templates", "wvc-theme"),
            "search_items" => __("Search Templates", "wvc-theme"),
            "parent_item_colon" => __("Parent Template:", "wvc-theme"),
            "not_found" => __("No templates found.", "wvc-theme"),
            "not_found_in_trash" => __("No templates found in Trash.", "wvc-theme"),
        );

        $args = array(
            "labels" => $labels,
            "public" => false,
            "publicly_queryable" => false,
            "show_ui" => true,
            "show_in_menu" => false,
            "query_var" => true,
            "rewrite" => array("slug" => "template-hierarchy"),
            "capability_type" => "post",
            "has_archive" => false,
            "hierarchical" => true, // Enable parent-child relationships
            "menu_position" => 26,
            "menu_icon" => "dashicons-networking",
            "supports" => array("title", "page-attributes"),
            "show_in_rest" => true,
            "rest_base" => "template-hierarchies",
            "rest_controller_class" => "WP_REST_Posts_Controller",
        );

        register_post_type("wvc_template", $args);
    }
    
    /**
     * Register meta fields for template
     */
    public function register_meta_fields()
    {
        // Template Type
        register_post_meta(
            "wvc_template", "_wvc_template_type", array(
            "type" => "string",
            "description" => "Template type (header, footer, content, sidebar, single)",
            "single" => true,
            "show_in_rest" => true,
            "default" => "content",
            "sanitize_callback" => array($this, "sanitize_template_type")
            )
        );
        
        // Version
        register_post_meta(
            "wvc_template", "_wvc_structure_version", array(
            "type" => "string",
            "description" => "Template version",
            "single" => true,
            "show_in_rest" => true,
            "default" => "1.0.0",
            "sanitize_callback" => "sanitize_text_field"
            )
        );
    }
    
    /**
     * Add meta boxes for template admin
     */
    public function add_meta_boxes()
    {
        add_meta_box(
            "wvc_template_structure",
            __("Template Configuration", "wvc-theme"),
            array($this, "template_structure_meta_box_callback"),
            "wvc_template",
            "normal",
            "high"
        );
        
        add_meta_box(
            "wvc_template_hierarchy_info",
            __("Hierarchy Information", "wvc-theme"),
            array($this, "hierarchy_info_meta_box_callback"),
            "wvc_template",
            "side",
            "default"
        );
    }
    
    /**
     * Template structure meta box callback
     */
    public function template_structure_meta_box_callback($post)
    {
        wp_nonce_field("wvc_save_template_hierarchy_meta", "wvc_template_hierarchy_nonce");
        
        $template_data = get_post_meta($post->ID, "_wvc_template", true);
        $template = $template_data ? json_decode($template_data, true) : array();
        
        echo "<div id=\"wvc-template-structure-editor\">";
        echo "<p><strong>" . __("Template Configuration", "wvc-theme") . "</strong></p>";
        echo "<textarea id=\"wvc_template\" name=\"wvc_template\" rows=\"15\" cols=\"100\" style=\"width: 100%; font-family: monospace;\">";
        echo esc_textarea(json_encode($template, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES));
        echo "</textarea>";
        echo "<p class=\"description\">" . __("Enter template configuration as JSON. Include template settings, content-managers structure, and any custom data.", "wvc-theme") . "</p>";
        echo "</div>";
        
        // Add JavaScript for JSON validation
        ?>
        <script>
        document.addEventListener("DOMContentLoaded", function() {
            const textarea = document.getElementById("wvc_template");
            if (textarea) {
                textarea.addEventListener("blur", function() {
                    try {
                        const parsed = JSON.parse(this.value || "{}");
                        this.style.borderColor = "#ddd";
                    } catch (e) {
                        this.style.borderColor = "#dc3232";
                    }
                });
            }
        });
        </script>
        <style>
        #wvc-template-structure-editor textarea {
            border: 1px solid #ddd;
            border-radius: 3px;
            padding: 10px;
        }
        </style>
        <?php
    }
    
    /**
     * Hierarchy information meta box callback
     */
    public function hierarchy_info_meta_box_callback($post)
    {
        $template_id = get_post_meta($post->ID, "_wvc_template_id", true);
        $hierarchy_level = get_post_meta($post->ID, "_wvc_hierarchy_level", true);
        
        echo "<div id=\"template-info-display\">";
        if ($template_id && isset($this->template_hierarchy[$template_id])) {
            $info = $this->template_hierarchy[$template_id];
            echo "<p><strong>" . __("Description:", "wvc-theme") . "</strong><br>" . esc_html($info["description"]) . "</p>";
            
            if ($info["parent"]) {
                echo "<p><strong>" . __("Parent Template:", "wvc-theme") . "</strong><br>" . esc_html($info["parent"]) . "</p>";
            }
            
            if ($hierarchy_level !== "") {
                echo "<p><strong>" . __("Hierarchy Level:", "wvc-theme") . "</strong><br>" . esc_html($hierarchy_level) . "</p>";
            }
            
            // Show children
            $children = $this->get_template_children($template_id);
            if (!empty($children)) {
                echo "<p><strong>" . __("Child Templates:", "wvc-theme") . "</strong><br>" . implode(", ", $children) . "</p>";
            }
        } else {
            echo "<p>" . __("Select a template type to see hierarchy information.", "wvc-theme") . "</p>";
        }
        echo "</div>";
    }
    
    /**
     * Save template meta
     */
    public function save_meta($post_id)
    {
        if (!isset($_POST["wvc_template_hierarchy_nonce"]) || !wp_verify_nonce($_POST["wvc_template_hierarchy_nonce"], "wvc_save_template_hierarchy_meta")) {
            return;
        }
        
        if (defined("DOING_AUTOSAVE") && DOING_AUTOSAVE) {
            return;
        }
        
        if (!current_user_can("edit_post", $post_id)) {
            return;
        }
        
        // Save template configuration
        if (isset($_POST["wvc_template"])) {
            $template_data = stripslashes($_POST["wvc_template"]);
            update_post_meta($post_id, "_wvc_template", $template_data);
        }
    }
    
    /**
     * Set template hierarchy relationships automatically
     */
    private function set_template_hierarchy($post_id, $template_id)
    {
        if (!isset($this->template_hierarchy[$template_id])) {
            return;
        }
        
        $template_info = $this->template_hierarchy[$template_id];
        $parent_template_type = $template_info["parent"];
        
        // Calculate hierarchy level
        $level = $this->calculate_hierarchy_level($template_id);
        update_post_meta($post_id, "_wvc_hierarchy_level", $level);
        
        // Find and set parent post if parent template type exists
        if ($parent_template_type) {
            $parent_post = $this->find_template_post_by_type($parent_template_type);
            if ($parent_post) {
                wp_update_post(array(
                    "ID" => $post_id,
                    "post_parent" => $parent_post->ID
                ));
            }
        } else {
            // No parent - this is a root template
            wp_update_post(array(
                "ID" => $post_id,
                "post_parent" => 0
            ));
        }
    }
    
    /**
     * Calculate hierarchy level for a template type
     */
    private function calculate_hierarchy_level($template_id, $visited = array())
    {
        if (in_array($template_id, $visited) || !isset($this->template_hierarchy[$template_id])) {
            return 0; // Prevent infinite loops or invalid templates
        }
        
        $parent = $this->template_hierarchy[$template_id]["parent"];
        if (!$parent) {
            return 0; // Root level
        }
        
        $visited[] = $template_id;
        return 1 + $this->calculate_hierarchy_level($parent, $visited);
    }
    
    /**
     * Find template post by template type
     */
    private function find_template_post_by_type($template_type)
    {
        $posts = get_posts(array(
            "post_type" => "wvc_template",
            "meta_query" => array(
                array(
                    "key" => "_wvc_template_id",
                    "value" => $template_type,
                    "compare" => "="
                )
            ),
            "posts_per_page" => 1,
            "post_status" => "any"
        ));
        
        return !empty($posts) ? $posts[0] : null;
    }
    
    /**
     * Get child template types for a given template type
     */
    private function get_template_children($template_type)
    {
        $children = array();
        foreach ($this->template_hierarchy as $type => $data) {
            if ($data["parent"] === $template_type) {
                $children[] = $type;
            }
        }
        return $children;
    }
    
    /**
     * Sanitize template ID
     */
    public function sanitize_template_id($value)
    {
        if (empty($value) || !array_key_exists($value, $this->template_hierarchy)) {
            return "";
        }
        return sanitize_text_field($value);
    }
    
    /**
     * Sanitize template data JSON
     */
    public function sanitize_template_data($value)
    {
        if (empty($value)) {
            return "";
        }
        
        $decoded = json_decode($value, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            return "";
        }
        
        // Basic sanitization of template data
        $sanitized = $this->sanitize_template_structure($decoded);
        
        return json_encode($sanitized, JSON_UNESCAPED_SLASHES);
    }
    
    /**
     * Sanitize template structure recursively
     */
    private function sanitize_template_structure($data)
    {
        if (is_array($data)) {
            $sanitized = array();
            foreach ($data as $key => $value) {
                $clean_key = sanitize_text_field($key);
                $sanitized[$clean_key] = $this->sanitize_template_structure($value);
            }
            return $sanitized;
        } elseif (is_string($data)) {
            return sanitize_text_field($data);
        } else {
            return $data; // Numbers, booleans, etc.
        }
    }
    
    /**
     * Sanitize template type
     */
    public function sanitize_template_type($value)
    {
        $valid_types = array("header", "footer", "content", "sidebar", "single");
        $sanitized_value = sanitize_text_field($value);
        
        if (in_array($sanitized_value, $valid_types)) {
            return $sanitized_value;
        }
        
        return "content"; // Default fallback
    }
    
    /**
     * Get all available WordPress template types
     */
    public function get_template_types()
    {
        return $this->template_hierarchy;
    }
    
    /**
     * Get available template component types
     */
    public function get_template_component_types()
    {
        return $this->template_types;
    }
} 