<?php
/**
 * Structure Validator
 *
 * This class provides validation functionality for template and page structures
 * in bulk operations. It validates data integrity, schema compliance, and
 * ensures proper formatting of imported content-managers.
 *
 * @package    WVC_Theme
 * @subpackage Bulk_Operations
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 * @link       https://github.com/your-repo/wvc-theme
 */

if (!defined("ABSPATH")) {
    exit;
}

class WVC_Structure_Validator
{
    
    /**
     * Validate complete structure
     */
    public static function validate_complete_structure($structure)
    {
        $errors = array();
        
        if (!is_array($structure)) {
            $errors[] = __("Structure must be an array", "wvc-theme");
            return $errors;
        }
        
        // Validate templates if present
        if (isset($structure["templates"]) && is_array($structure["templates"])) {
            $template_errors = self::validate_templates($structure["templates"]);
            if (!empty($template_errors)) {
                $errors["templates"] = $template_errors;
            }
        }
        
        // Validate pages if present
        if (isset($structure["pages"]) && is_array($structure["pages"])) {
            $page_errors = self::validate_pages($structure["pages"]);
            if (!empty($page_errors)) {
                $errors["pages"] = $page_errors;
            }
        }
        
        // Validate posts if present
        if (isset($structure["posts"]) && is_array($structure["posts"])) {
            $post_errors = self::validate_posts($structure["posts"]);
            if (!empty($post_errors)) {
                $errors["posts"] = $post_errors;
            }
        }
        
        return $errors;
    }
    
    /**
     * Validate templates array
     */
    public static function validate_templates($templates)
    {
        $errors = array();
        
        if (!is_array($templates)) {
            return array(__("Templates must be an array", "wvc-theme"));
        }
        
        foreach ($templates as $index => $template) {
            $template_errors = self::validate_single_template($template);
            if (!empty($template_errors)) {
                $errors[$index] = $template_errors;
            }
        }
        
        return $errors;
    }
    
    /**
     * Validate single template
     */
    public static function validate_single_template($template)
    {
        $errors = array();
        $required_fields = array("name", "file_path");
        
        if (!is_array($template)) {
            return array(__("Template must be an object", "wvc-theme"));
        }
        
        // Check required fields
        foreach ($required_fields as $field) {
            if (!isset($template[$field]) || empty($template[$field])) {
                $errors[] = sprintf(__("Missing required field: %s", "wvc-theme"), $field);
            }
        }
        
        
        // Validate hierarchy level
        if (isset($template["hierarchy_level"]) && !is_numeric($template["hierarchy_level"])) {
            $errors[] = __("Hierarchy level must be a number", "wvc-theme");
        }
        
        // Validate template type
        if (isset($template["template_type"])) {
            $allowed_types = array("page", "post", "archive", "single", "index", "custom");
            if (!in_array($template["template_type"], $allowed_types)) {
                $errors[] = sprintf(__("Invalid template type. Allowed: %s", "wvc-theme"), implode(", ", $allowed_types));
            }
        }
        
        return $errors;
    }
    
    /**
     * Validate pages array
     */
    public static function validate_pages($pages)
    {
        $errors = array();
        
        if (!is_array($pages)) {
            return array(__("Pages must be an array", "wvc-theme"));
        }
        
        $slugs = array();
        foreach ($pages as $index => $page) {
            $page_errors = self::validate_single_page($page);
            
            // Check for duplicate slugs
            if (isset($page["slug"])) {
                if (in_array($page["slug"], $slugs)) {
                    $page_errors[] = __("Duplicate page slug found", "wvc-theme");
                } else {
                    $slugs[] = $page["slug"];
                }
            }
            
            if (!empty($page_errors)) {
                $errors[$index] = $page_errors;
            }
        }
        
        return $errors;
    }
    
    /**
     * Validate single page
     */
    public static function validate_single_page($page)
    {
        $errors = array();
        $required_fields = array("title", "slug");
        
        if (!is_array($page)) {
            return array(__("Page must be an object", "wvc-theme"));
        }
        
        // Check required fields
        foreach ($required_fields as $field) {
            if (!isset($page[$field]) || empty($page[$field])) {
                $errors[] = sprintf(__("Missing required field: %s", "wvc-theme"), $field);
            }
        }
        
        // Validate slug format
        if (isset($page["slug"])) {
            if (!self::is_valid_slug($page["slug"])) {
                $errors[] = __("Invalid slug format", "wvc-theme");
            }
        }
        
        // Validate status
        if (isset($page["status"])) {
            $allowed_statuses = array("publish", "draft", "private", "pending");
            if (!in_array($page["status"], $allowed_statuses)) {
                $errors[] = sprintf(__("Invalid status. Allowed: %s", "wvc-theme"), implode(", ", $allowed_statuses));
            }
        }
        
        // Validate menu order
        if (isset($page["menu_order"]) && !is_numeric($page["menu_order"])) {
            $errors[] = __("Menu order must be a number", "wvc-theme");
        }
        
        // Validate meta data
        if (isset($page["meta_data"]) && !is_array($page["meta_data"])) {
            $errors[] = __("Meta data must be an object", "wvc-theme");
        }
        
        return $errors;
    }
    
    /**
     * Validate posts array
     */
    public static function validate_posts($posts)
    {
        $errors = array();
        
        if (!is_array($posts)) {
            return array(__("Posts must be an array", "wvc-theme"));
        }
        
        $slugs = array();
        foreach ($posts as $index => $post) {
            $post_errors = self::validate_single_post($post);
            
            // Check for duplicate slugs
            if (isset($post["slug"]) && !empty($post["slug"])) {
                if (in_array($post["slug"], $slugs)) {
                    $post_errors[] = __("Duplicate post slug found", "wvc-theme");
                } else {
                    $slugs[] = $post["slug"];
                }
            }
            
            if (!empty($post_errors)) {
                $errors[$index] = $post_errors;
            }
        }
        
        return $errors;
    }
    
    /**
     * Validate single post
     */
    public static function validate_single_post($post)
    {
        $errors = array();
        $required_fields = array("title", "content-managers");
        
        if (!is_array($post)) {
            return array(__("Post must be an object", "wvc-theme"));
        }
        
        // Check required fields
        foreach ($required_fields as $field) {
            if (!isset($post[$field]) || empty($post[$field])) {
                $errors[] = sprintf(__("Missing required field: %s", "wvc-theme"), $field);
            }
        }
        
        // Validate slug format if provided
        if (isset($post["slug"]) && !empty($post["slug"])) {
            if (!self::is_valid_slug($post["slug"])) {
                $errors[] = __("Invalid slug format", "wvc-theme");
            }
        }
        
        // Validate status
        if (isset($post["status"])) {
            $allowed_statuses = array("publish", "draft", "private", "pending");
            if (!in_array($post["status"], $allowed_statuses)) {
                $errors[] = sprintf(__("Invalid status. Allowed: %s", "wvc-theme"), implode(", ", $allowed_statuses));
            }
        }
        
        // Validate categories
        if (isset($post["categories"]) && !is_array($post["categories"])) {
            $errors[] = __("Categories must be an array", "wvc-theme");
        }
        
        // Validate tags
        if (isset($post["tags"]) && !is_array($post["tags"])) {
            $errors[] = __("Tags must be an array", "wvc-theme");
        }
        
        // Validate meta data
        if (isset($post["meta_data"]) && !is_array($post["meta_data"])) {
            $errors[] = __("Meta data must be an object", "wvc-theme");
        }
        
        // Validate featured image URL
        if (isset($post["featured_image"]) && !empty($post["featured_image"])) {
            if (!filter_var($post["featured_image"], FILTER_VALIDATE_URL)) {
                $errors[] = __("Invalid featured image URL", "wvc-theme");
            }
        }
        
        return $errors;
    }
    
    /**
     * Check if template path is valid
     */
    private static function is_valid_template_path($path)
    {
        // Check for basic PHP file extension and no directory traversal
        return preg_match("/^[a-zA-Z0-9\/_-]+\.php$/", $path) && strpos($path, "..") === false;
    }
    
    /**
     * Check if slug is valid
     */
    private static function is_valid_slug($slug)
    {
        // WordPress slug format: lowercase, hyphens, numbers, letters
        return preg_match("/^[a-z0-9\-]+$/", $slug);
    }
    
    /**
     * Validate hierarchical relationships
     */
    public static function validate_hierarchy($items, $type = "pages")
    {
        $errors = array();
        $items_by_slug = array();
        
        // Create lookup array
        foreach ($items as $item) {
            if (isset($item["slug"])) {
                $items_by_slug[$item["slug"]] = $item;
            }
        }
        
        // Check parent relationships
        foreach ($items as $index => $item) {
            $parent_field = ($type === "pages") ? "parent_page" : "parent_template";
            
            if (isset($item[$parent_field]) && !empty($item[$parent_field])) {
                // Check if parent exists
                if (!isset($items_by_slug[$item[$parent_field]])) {
                    $errors[$index][] = sprintf(__("Parent %s not found: %s", "wvc-theme"), $type, $item[$parent_field]);
                }
                
                // Check for circular references
                if (self::has_circular_reference($item, $items_by_slug, $parent_field)) {
                    $errors[$index][] = __("Circular reference detected in hierarchy", "wvc-theme");
                }
            }
        }
        
        return $errors;
    }
    
    /**
     * Check for circular references in hierarchy
     */
    private static function has_circular_reference($item, $items_by_slug, $parent_field, $visited = array())
    {
        if (!isset($item["slug"])) {
            return false;
        }
        
        if (in_array($item["slug"], $visited)) {
            return true; // Circular reference found
        }
        
        $visited[] = $item["slug"];
        
        if (isset($item[$parent_field]) && !empty($item[$parent_field])) {
            if (isset($items_by_slug[$item[$parent_field]])) {
                return self::has_circular_reference(
                    $items_by_slug[$item[$parent_field]], 
                    $items_by_slug, 
                    $parent_field, 
                    $visited
                );
            }
        }
        
        return false;
    }
    
    /**
     * Load and parse existing YAML template configuration
     * 
     * @param string $yaml_path Path to template configuration YAML file
     * @return array|false Parsed YAML structure or false on failure
     */
    public static function load_template_yaml($yaml_path = null)
    {
        // Default YAML path - adjust to your existing file location
        if (!$yaml_path) {
            $yaml_path = get_template_directory() . "/template-config.yaml";
        }
        
        if (!file_exists($yaml_path)) {
            error_log("WVC: Template YAML file not found at: " . $yaml_path);
            return false;
        }
        
        // Parse YAML file
        $config = self::parse_yaml_file($yaml_path);
        if (!$config) {
            error_log("WVC: Failed to parse template YAML");
            return false;
        }
        
        return $config;
    }
    
    /**
     * Map submitted templates to YAML schema and validate
     * 
     * @param array $submitted_templates Templates submitted via REST
     * @param string $yaml_path Path to template YAML file
     * @return array Result with mapped and validated templates
     */
    public static function validate_templates_against_yaml($submitted_templates, $yaml_path = null)
    {
        $result = array(
            "valid_templates" => array(),
            "invalid_templates" => array(),
            "unknown_templates" => array(),
            "missing_required_templates" => array()
        );
        
        // Load YAML config
        $config = self::load_template_yaml($yaml_path);
        if (!$config) {
            $result["invalid_templates"]["yaml_error"] = __("Could not load template YAML configuration", "wvc-theme");
            return $result;
        }
        
        // Create lookup for YAML templates
        $yaml_templates_by_slug = array();
        if (isset($config["templates"]) && is_array($config["templates"])) {
            foreach ($config["templates"] as $yaml_template) {
                if (isset($yaml_template["slug"])) {
                    $yaml_templates_by_slug[$yaml_template["slug"]] = $yaml_template;
                }
            }
        }
        
        // Track which YAML templates were found
        $found_yaml_templates = array();
        
        // Process each submitted template
        foreach ($submitted_templates as $index => $template) {
            $template_slug = $template["slug"] ?? "";
            
            if (empty($template_slug)) {
                $result["invalid_templates"][$index] = array(__("Template missing slug", "wvc-theme"));
                continue;
            }
            
            // Check if template exists in YAML
            if (isset($yaml_templates_by_slug[$template_slug])) {
                $yaml_template = $yaml_templates_by_slug[$template_slug];
                $found_yaml_templates[] = $template_slug;
                
                // Validate against YAML schema
                $validation_errors = self::validate_template_against_yaml($template, $yaml_template);
                
                if (empty($validation_errors)) {
                    $result["valid_templates"][] = array_merge($template, array(
                        "yaml_config" => $yaml_template
                    ));
                } else {
                    $result["invalid_templates"][$template_slug] = $validation_errors;
                }
            } else {
                // Template not found in YAML - unknown template
                $result["unknown_templates"][] = $template;
            }
        }
        
        // Find missing required templates
        foreach ($yaml_templates_by_slug as $slug => $yaml_template) {
            if (!in_array($slug, $found_yaml_templates)) {
                if (isset($yaml_template["required"]) && $yaml_template["required"]) {
                    $result["missing_required_templates"][] = $yaml_template;
                }
            }
        }
        
        return $result;
    }
    
    /**
     * Resolve template for current request based on template priority
     * 
     * @param string $request_type Type of request (e.g., "shop", "product", "category")
     * @param array $context Additional context for template resolution
     * @return WP_Post|false Template post object to use or false if none found
     */
    public static function resolve_template_for_request($request_type, $context = array())
    {
        // Build template priority list based on request type
        $template_priority = self::build_template_priority($request_type, $context);
        
        // Check each template in priority order
        foreach ($template_priority as $template_name) {
            $template_post = self::get_template_by_title($template_name);
            if ($template_post) {
                return $template_post;
            }
        }
        
        return false; // No template found
    }
    
    /**
     * Build template priority list for a request type
     * 
     * @param string $request_type Type of request
     * @param array $context Additional context
     * @return array Array of template names in priority order
     */
    private static function build_template_priority($request_type, $context = array())
    {
        $priority = array();
        
        // Get post type and other context
        $post_type = $context["post_type"] ?? "page";
        $is_single = $context["is_single"] ?? false;
        $is_archive = $context["is_archive"] ?? false;
        
        if ($is_single) {
            // Single post/page templates
            $priority[] = ucwords(str_replace("-", " ", "{$post_type}-{$request_type}")); // e.g., "Page Shop"
            $priority[] = ucwords(str_replace("-", " ", "single-{$post_type}"));          // e.g., "Single Page"
            $priority[] = "Singular";                                                    // Generic singular
        } elseif ($is_archive) {
            // Archive templates
            $priority[] = ucwords(str_replace("-", " ", "archive-{$request_type}"));     // e.g., "Archive Product"
            $priority[] = ucwords(str_replace("-", " ", "archive-{$post_type}"));        // e.g., "Archive Page"
            $priority[] = "Archive";                                                     // Generic archive
        } else {
            // Regular page templates
            $priority[] = ucwords(str_replace("-", " ", "page-{$request_type}"));        // e.g., "Page Shop"
            $priority[] = "Page";                                                        // Generic page
        }
        
        // Always fallback to index
        $priority[] = "Index";
        
        return array_unique($priority);
    }
    
    /**
     * Build template name from slug (converts slug to title format)
     * 
     * @param string $template_slug Template slug (e.g., "page-shop")
     * @return string Template title (e.g., "Page Shop")
     */
    public static function slug_to_template_title($template_slug)
    {
        return ucwords(str_replace(array("-", "_"), " ", $template_slug));
    }
    
    /**
     * Build template slug from title (converts title to slug format)
     * 
     * @param string $template_title Template title (e.g., "Page Shop")
     * @return string Template slug (e.g., "page-shop")
     */
    public static function title_to_template_slug($template_title)
    {
        return strtolower(str_replace(" ", "-", trim($template_title)));
    }
    
    /**
     * Check if a template exists in WordPress by title
     * 
     * @param string $template_name Template name/title to check
     * @return bool True if template exists
     */
    private static function template_exists($template_name)
    {
        $template_posts = get_posts(array(
            "post_type" => "wvc_template", // Adjust to your CPT name
            "title" => $template_name,
            "post_status" => array("publish", "draft"),
            "numberposts" => 1
        ));
        
        return !empty($template_posts);
    }
    
    /**
     * Check if template title is unique
     * 
     * @param string $template_title Template title to check
     * @param int $exclude_post_id Post ID to exclude from check (for updates)
     * @return bool True if title is unique
     */
    public static function is_template_title_unique($template_title, $exclude_post_id = 0)
    {
        if (empty($template_title)) {
            return false;
        }
        
        $args = array(
            "post_type" => "wvc_template",
            "title" => $template_title,
            "post_status" => array("publish", "draft", "private", "pending"),
            "numberposts" => 1,
            "fields" => "ids"
        );
        
        if ($exclude_post_id > 0) {
            $args["post__not_in"] = array($exclude_post_id);
        }
        
        $existing_posts = get_posts($args);
        
        return empty($existing_posts);
    }
    
    /**
     * Validate template titles for uniqueness
     * 
     * @param array $templates Array of templates to validate
     * @return array Validation results with duplicates identified
     */
    public static function validate_template_title_uniqueness($templates)
    {
        $result = array(
            "valid_templates" => array(),
            "duplicate_titles" => array(),
            "invalid_titles" => array()
        );
        
        $titles_seen = array();
        $titles_in_db = array();
        
        foreach ($templates as $index => $template) {
            $title = $template["title"] ?? $template["name"] ?? "";
            
            if (empty($title)) {
                $result["invalid_titles"][$index] = __("Template missing title/name", "wvc-theme");
                continue;
            }
            
            // Check for duplicates within submitted templates
            if (isset($titles_seen[$title])) {
                $result["duplicate_titles"][$index] = sprintf(
                    __("Duplicate title found in submission: %s (also at index %d)", "wvc-theme"), 
                    $title, 
                    $titles_seen[$title]
                );
                continue;
            }
            
            // Check for existing templates in database
            if (!self::is_template_title_unique($title)) {
                $result["duplicate_titles"][$index] = sprintf(
                    __("Template title already exists in database: %s", "wvc-theme"), 
                    $title
                );
                continue;
            }
            
            // Mark as seen and valid
            $titles_seen[$title] = $index;
            $result["valid_templates"][] = $template;
        }
        
        return $result;
    }
    
    /**
     * Get template by title
     * 
     * @param string $template_title Template title to find
     * @return WP_Post|null Template post object or null if not found
     */
    public static function get_template_by_title($template_title)
    {
        if (empty($template_title)) {
            return null;
        }
        
        $template_posts = get_posts(array(
            "post_type" => "wvc_template",
            "title" => $template_title,
            "post_status" => array("publish", "draft"),
            "numberposts" => 1
        ));
        
        return !empty($template_posts) ? $template_posts[0] : null;
    }
    
    /**
     * Validate template against YAML configuration
     */
    private static function validate_template_against_yaml($template, $yaml_config)
    {
        $errors = array();
        
        // Check required fields from YAML
        if (isset($yaml_config["required_fields"]) && is_array($yaml_config["required_fields"])) {
            foreach ($yaml_config["required_fields"] as $field) {
                if (!isset($template[$field]) || empty($template[$field])) {
                    $errors[] = sprintf(__("Missing required field: %s", "wvc-theme"), $field);
                }
            }
        }
        
        // Check template type if specified
        if (isset($yaml_config["template_type"]) && isset($template["template_type"])) {
            if ($template["template_type"] !== $yaml_config["template_type"]) {
                $errors[] = sprintf(__("Template type mismatch. Expected: %s, Got: %s", "wvc-theme"), 
                    $yaml_config["template_type"], $template["template_type"]);
            }
        }
        
        // Check allowed types if specified
        if (isset($yaml_config["allowed_types"]) && is_array($yaml_config["allowed_types"])) {
            $template_type = $template["template_type"] ?? "";
            if (!empty($template_type) && !in_array($template_type, $yaml_config["allowed_types"])) {
                $errors[] = sprintf(__("Template type not allowed: %s", "wvc-theme"), $template_type);
            }
        }
        
        return $errors;
    }
    
    /**
     * Parse YAML file (simple implementation)
     */
    private static function parse_yaml_file($file_path)
    {
        if (!file_exists($file_path)) {
            return false;
        }
        
        $content = file_get_contents($file_path);
        if ($content === false) {
            return false;
        }
        
        return self::simple_yaml_parse($content);
    }
    
    /**
     * Simple YAML parser for template hierarchy
     */
    private static function simple_yaml_parse($yaml_content)
    {
        $lines = explode("\n", $yaml_content);
        $result = array();
        $current_section = null;
        $current_template = null;
        $indent_level = 0;
        
        foreach ($lines as $line) {
            $original_line = $line;
            $line = rtrim($line);
            
            if (empty($line) || strpos(trim($line), "#") === 0) continue;
            
            // Calculate indentation
            $current_indent = strlen($line) - strlen(ltrim($line));
            $line = trim($line);
            
            if (strpos($line, "templates:") === 0) {
                $current_section = "templates";
                $result["templates"] = array();
                $indent_level = $current_indent;
            } elseif ($current_section === "templates" && strpos($line, "- slug:") === 0) {
                if ($current_template !== null) {
                    $result["templates"][] = $current_template;
                }
                $current_template = array("slug" => trim(str_replace("- slug:", "", $line), ' "\''));
            } elseif ($current_template !== null && $current_indent > $indent_level) {
                $parts = explode(":", $line, 2);
                if (count($parts) === 2) {
                    $key = trim($parts[0]);
                    $value = trim($parts[1], ' "\'');
                    
                    // Handle different value types
                    if ($value === "true") {
                        $value = true;
                    } elseif ($value === "false") {
                        $value = false;
                    } elseif (is_numeric($value)) {
                        $value = (int)$value;
                    } elseif (strpos($value, "[") === 0 && strpos($value, "]") !== false) {
                        // Simple array parsing
                        $value = str_replace(array("[", "]", '"', "'"), "", $value);
                        $value = array_map("trim", explode(",", $value));
                    }
                    
                    $current_template[$key] = $value;
                }
            }
        }
        
        if ($current_template !== null) {
            $result["templates"][] = $current_template;
        }
        
        return $result;
    }
    
    /**
     * WordPress hook to prevent duplicate template titles
     * Call this in your theme's functions.php or plugin activation
     */
    public static function init_wordpress_hooks()
    {
        add_action("wp_insert_post", array(__CLASS__, "prevent_duplicate_template_titles"), 10, 3);
        add_filter("wp_insert_post_data", array(__CLASS__, "validate_template_title_before_save"), 10, 2);
    }
    
    /**
     * Prevent duplicate template titles during post insert/update
     * 
     * @param int $post_id Post ID
     * @param WP_Post $post Post object
     * @param bool $update Whether this is an update
     */
    public static function prevent_duplicate_template_titles($post_id, $post, $update)
    {
        // Only check for wvc_template post type
        if ($post->post_type !== "wvc_template") {
            return;
        }
        
        // Skip if this is an auto-save or revision
        if (wp_is_post_autosave($post_id) || wp_is_post_revision($post_id)) {
            return;
        }
        
        // Check for duplicate title
        if (!self::is_template_title_unique($post->post_title, $post_id)) {
            // Prevent the save by updating the post with an error
            wp_update_post(array(
                "ID" => $post_id,
                "post_status" => "draft",
                "post_title" => $post->post_title . " (Duplicate - " . date("Y-m-d H:i:s") . ")"
            ));
            
            // Add admin notice
            add_action("admin_notices", function() use ($post) {
                echo '<div class="notice notice-error"><p>';
                printf(__("Template title \"%s\" already exists. Title has been modified to prevent duplicates.", "wvc-theme"), 
                    esc_html($post->post_title));
                echo '</p></div>';
            });
        }
    }
    
    /**
     * Validate template title before saving
     * 
     * @param array $data Post data array
     * @param array $postarr Post array from form
     * @return array Modified post data
     */
    public static function validate_template_title_before_save($data, $postarr)
    {
        // Only check for wvc_template post type
        if ($data["post_type"] !== "wvc_template") {
            return $data;
        }
        
        // Skip if this is an auto-save or revision
        if (defined("DOING_AUTOSAVE") && DOING_AUTOSAVE) {
            return $data;
        }
        
        $post_id = $postarr["ID"] ?? 0;
        $title = $data["post_title"];
        
        // Check for duplicate title
        if (!empty($title) && !self::is_template_title_unique($title, $post_id)) {
            // Modify title to make it unique
            $counter = 1;
            $original_title = $title;
            
            while (!self::is_template_title_unique($title, $post_id)) {
                $title = $original_title . " (" . $counter . ")";
                $counter++;
                
                // Prevent infinite loop
                if ($counter > 100) {
                    $title = $original_title . " (" . uniqid() . ")";
                    break;
                }
            }
            
            $data["post_title"] = $title;
            
            // Store original title as meta for reference
            add_action("save_post", function($saved_post_id) use ($original_title, $post_id) {
                if ($saved_post_id === $post_id) {
                    update_post_meta($saved_post_id, "_original_duplicate_title", $original_title);
                }
            });
        }
        
        return $data;
    }
} 