(function (global, factory) {
    typeof exports === 'object' && typeof module !== 'undefined' ? factory(exports) :
    typeof define === 'function' && define.amd ? define(['exports'], factory) :
    (global = typeof globalThis !== 'undefined' ? globalThis : global || self, factory(global.WvcUtils = {}));
})(this, (function (exports) { 'use strict';

    class WPClient {
        constructor(config) {
            this.baseUrl = config.baseUrl.replace(/\/$/, "");
            this.auth = config.auth;
            this.cache = config.cache;
        }
        buildUrl(endpoint, params, namespace = "wp/v2") {
            const searchParams = new URLSearchParams();
            // Set the REST route parameter with custom namespace
            searchParams.set("rest_route", `/${namespace}/${endpoint}`);
            // Add other parameters if they exist
            if (params) {
                for (const [key, value] of Object.entries(params)) {
                    if (value !== undefined && value !== null) {
                        if (Array.isArray(value)) {
                            value.forEach((item) => {
                                searchParams.append(key, item.toString());
                            });
                        }
                        else {
                            searchParams.set(key, value.toString());
                        }
                    }
                }
            }
            return `${this.baseUrl}?${searchParams.toString()}`;
        }
        getHeaders() {
            const headers = {
                "Content-Type": "application/json"
            };
            if (this.auth) {
                if ("username" in this.auth && "password" in this.auth) {
                    const credentials = btoa(`${this.auth.username}:${this.auth.password}`);
                    headers["Authorization"] = `Basic ${credentials}`;
                }
                else if ("token" in this.auth) {
                    headers["Authorization"] = `Bearer ${this.auth.token}`;
                }
            }
            return headers;
        }
        async handleResponse(response) {
            const headers = {};
            response.headers.forEach((value, key) => {
                headers[key] = value;
            });
            if (!response.ok) {
                let error;
                try {
                    const errorData = await response.json();
                    error = {
                        code: errorData.code || "unknown_error",
                        message: errorData.message || "An unknown error occurred",
                        data: errorData.data || { status: response.status }
                    };
                }
                catch (_a) {
                    error = {
                        code: "http_error",
                        message: `HTTP ${response.status}: ${response.statusText}`,
                        data: { status: response.status }
                    };
                }
                throw error;
            }
            const data = await response.json();
            return {
                data,
                headers,
                status: response.status
            };
        }
        async request(endpoint, params, options = {}) {
            const url = this.buildUrl(endpoint, params);
            const config = {
                ...options,
                headers: {
                    ...this.getHeaders(),
                    ...options.headers
                }
            };
            const response = await fetch(url, config);
            return this.handleResponse(response);
        }
        async get(endpoint, params, flushCache = false) {
            const methodName = `get_${endpoint}`;
            const args = [endpoint, params];
            // Check cache first (unless flushCache is true)
            if (!flushCache && this.cache) {
                const cachedData = this.cache.get(methodName, args);
                if (cachedData) {
                    return cachedData;
                }
            }
            // Make request
            const response = await this.request(endpoint, params, { method: "GET" });
            // Store in cache (only cache successful responses)
            if (response.status >= 200 && response.status < 300 && this.cache) {
                this.cache.set(methodName, args, response);
            }
            return response;
        }
        /**
         * Make a GET request with a custom namespace
         * @param endpoint - The endpoint path
         * @param params - Query parameters
         * @param namespace - Custom namespace (default: wp/v2)
         * @param flushCache - Whether to flush cache for this request
         * @returns Promise<WPResponse<T>>
         */
        async getWithNamespace(endpoint, params, namespace = "wp/v2", flushCache = false) {
            const url = this.buildUrl(endpoint, params, namespace);
            const methodName = `getWithNamespace_${namespace}_${endpoint}`;
            const args = [endpoint, params, namespace];
            // Check cache first (unless flushCache is true)
            if (!flushCache && this.cache) {
                const cachedData = this.cache.get(methodName, args);
                if (cachedData) {
                    return cachedData;
                }
            }
            const config = {
                method: "GET",
                headers: this.getHeaders()
            };
            const response = await fetch(url, config);
            const result = await this.handleResponse(response);
            // Store in cache (only cache successful responses)
            if (result.status >= 200 && result.status < 300 && this.cache) {
                this.cache.set(methodName, args, result);
            }
            return result;
        }
        /**
         * Make a POST request with a custom namespace
         * @param endpoint - The endpoint path
         * @param data - Request body data
         * @param params - Query parameters
         * @param namespace - Custom namespace (default: wp/v2)
         * @returns Promise<WPResponse<T>>
         */
        async postWithNamespace(endpoint, data, params, namespace = "wp/v2") {
            const url = this.buildUrl(endpoint, params, namespace);
            const config = {
                method: "POST",
                headers: this.getHeaders(),
                body: JSON.stringify(data)
            };
            const response = await fetch(url, config);
            const result = await this.handleResponse(response);
            return result;
        }
        async post(endpoint, data, params) {
            const response = await this.request(endpoint, params, {
                method: "POST",
                body: JSON.stringify(data)
            });
            return response;
        }
        async put(endpoint, data, params) {
            const response = await this.request(endpoint, params, {
                method: "PUT",
                body: JSON.stringify(data)
            });
            return response;
        }
        async delete(endpoint, params) {
            const response = await this.request(endpoint, params, { method: "DELETE" });
            return response;
        }
        /**
         * Get cache statistics
         * @returns Cache statistics
         */
        getCacheStats() {
            var _a;
            return ((_a = this.cache) === null || _a === void 0 ? void 0 : _a.getStats()) || null;
        }
        /**
         * Clear all cache entries
         */
        clearCache() {
            var _a;
            (_a = this.cache) === null || _a === void 0 ? void 0 : _a.clear();
        }
        /**
         * Flush expired cache entries
         */
        flushCache() {
            var _a;
            (_a = this.cache) === null || _a === void 0 ? void 0 : _a.flushExpired();
        }
        /**
         * Clear cache entries by method name pattern
         * @param methodPattern - Method name pattern (regex string)
         * @returns Number of entries cleared
         */
        clearCacheByMethod(methodPattern) {
            var _a;
            return ((_a = this.cache) === null || _a === void 0 ? void 0 : _a.clearByMethod(methodPattern)) || 0;
        }
        /**
         * Flush cache for a specific method and arguments
         * @param methodName - Method name
         * @param args - Method arguments
         */
        flushCacheForMethod(methodName, args = []) {
            var _a;
            (_a = this.cache) === null || _a === void 0 ? void 0 : _a.flush(methodName, args);
        }
        /**
         * Enable cache
         */
        enableCache() {
            var _a;
            (_a = this.cache) === null || _a === void 0 ? void 0 : _a.enable();
        }
        /**
         * Disable cache and clear all entries
         */
        disableCache() {
            var _a;
            (_a = this.cache) === null || _a === void 0 ? void 0 : _a.disable();
        }
        /**
         * Check if cache is enabled
         * @returns True if cache is enabled
         */
        isCacheEnabled() {
            var _a;
            return ((_a = this.cache) === null || _a === void 0 ? void 0 : _a.isEnabled()) || false;
        }
        /**
         * Update cache configuration
         * @param config - New cache configuration
         */
        updateCacheConfig(config) {
            var _a;
            (_a = this.cache) === null || _a === void 0 ? void 0 : _a.updateConfig(config);
        }
        /**
         * Get cache configuration
         * @returns Current cache configuration
         */
        getCacheConfig() {
            var _a;
            return ((_a = this.cache) === null || _a === void 0 ? void 0 : _a.getConfig()) || null;
        }
        /**
         * Get cache entries for debugging
         * @returns Array of cache entries
         */
        getCacheEntries() {
            var _a;
            return ((_a = this.cache) === null || _a === void 0 ? void 0 : _a.getEntries()) || [];
        }
        /**
         * Get cache entries for a specific method
         * @param methodName - Method name to filter by
         * @returns Array of matching cache entries
         */
        getCacheEntriesByMethod(methodName) {
            var _a;
            return ((_a = this.cache) === null || _a === void 0 ? void 0 : _a.getEntriesByMethod(methodName)) || [];
        }
    }

    /**
     * WordPress Internal Names → REST API Endpoint Mappings
     *
     * This file centralizes the mapping between WordPress internal names (used in queries)
     * and the corresponding REST API endpoint names.
     */
    /**
     * Map WordPress taxonomy names to REST API endpoint names
     *
     * WordPress uses internal taxonomy names in queries (like tax_query),
     * but the REST API endpoints use different names (usually plural).
     */
    const TAXONOMY_MAPPINGS = {
        // Core WordPress taxonomies
        'category': 'categories',
        'post_tag': 'tags',
        'nav_menu': 'nav-menus',
        // Common plugin taxonomies (examples - uncomment and modify as needed)
        // 'product_cat': 'product-categories',      // WooCommerce
        // 'product_tag': 'product-tags',            // WooCommerce
        // 'product_type': 'product-types',          // WooCommerce
        // 'event_category': 'event-categories',     // Events plugins
        // 'portfolio_category': 'portfolio-categories', // Portfolio plugins
        // Add your custom taxonomies here:
        // 'my_taxonomy': 'my-taxonomy-endpoint',
    };
    /**
     * Map WordPress post type names to REST API endpoint names
     *
     * WordPress uses internal post type names in queries,
     * but the REST API endpoints may use different names.
     */
    const POST_TYPE_MAPPINGS = {
        // Core WordPress post types
        'post': 'posts',
        'page': 'pages',
        'attachment': 'media',
        // Common plugin post types (examples - uncomment and modify as needed)
        // 'product': 'products',                    // WooCommerce
        // 'shop_order': 'orders',                   // WooCommerce
        // 'shop_coupon': 'coupons',                 // WooCommerce
        // 'event': 'events',                        // Events plugins
        // 'portfolio': 'portfolio',                 // Portfolio plugins
        // 'testimonial': 'testimonials',            // Testimonials plugins
        // 'faq': 'faqs',                           // FAQ plugins
        // Add your custom post types here:
        // 'my_post_type': 'my-post-type-endpoint',
    };
    /**
     * Map WordPress taxonomy name to REST API endpoint name
     * @param taxonomy - WordPress taxonomy name (e.g., 'nav_menu', 'category')
     * @returns REST API endpoint name (e.g., 'nav-menus', 'categories')
     */
    function mapTaxonomyToEndpoint(taxonomy) {
        return TAXONOMY_MAPPINGS[taxonomy] || taxonomy;
    }
    /**
     * Map WordPress post type name to REST API endpoint name
     * @param postType - WordPress post type name (e.g., 'post', 'page')
     * @returns REST API endpoint name (e.g., 'posts', 'pages')
     */
    function mapPostTypeToEndpoint(postType) {
        return POST_TYPE_MAPPINGS[postType] || postType;
    }
    /**
     * Get all registered taxonomy mappings
     * @returns Object with all taxonomy mappings
     */
    function getAllTaxonomyMappings() {
        return { ...TAXONOMY_MAPPINGS };
    }
    /**
     * Get all registered post type mappings
     * @returns Object with all post type mappings
     */
    function getAllPostTypeMappings() {
        return { ...POST_TYPE_MAPPINGS };
    }
    /**
     * Register a new taxonomy mapping
     * @param taxonomy - WordPress taxonomy name
     * @param endpoint - REST API endpoint name
     */
    function registerTaxonomyMapping(taxonomy, endpoint) {
        TAXONOMY_MAPPINGS[taxonomy] = endpoint;
    }
    /**
     * Register a new post type mapping
     * @param postType - WordPress post type name
     * @param endpoint - REST API endpoint name
     */
    function registerPostTypeMapping(postType, endpoint) {
        POST_TYPE_MAPPINGS[postType] = endpoint;
    }
    /**
     * Check if a taxonomy mapping exists
     * @param taxonomy - WordPress taxonomy name
     * @returns True if mapping exists
     */
    function hasTaxonomyMapping(taxonomy) {
        return taxonomy in TAXONOMY_MAPPINGS;
    }
    /**
     * Check if a post type mapping exists
     * @param postType - WordPress post type name
     * @returns True if mapping exists
     */
    function hasPostTypeMapping(postType) {
        return postType in POST_TYPE_MAPPINGS;
    }

    /**
     * WordPress Posts API
     * Simplified to core functions that handle everything through parameters
     */
    class WPPostsAPI {
        constructor(client) {
            this.client = client;
        }
        /**
         * Get posts with WordPress-style query parameters
         * This is the main function that handles all post queries
         * @param params - WordPress query parameters (similar to WP_Query)
         * @returns Promise<WPResponse<WPPost[]>> - Includes posts data and response headers (e.g., x-wp-total, x-wp-totalpages)
         */
        async get_posts(params = {}) {
            const response = await this.client.get('posts', this.transformQueryParams(params));
            return response;
        }
        /**
         * Get pages with WordPress-style query parameters
         * @param params - WordPress query parameters (similar to WP_Query)
         * @returns Promise<WPResponse<WPPage[]>> - Includes pages data and response headers (e.g., x-wp-total, x-wp-totalpages)
         */
        async get_pages(params = {}) {
            const response = await this.client.get('pages', this.transformQueryParams(params));
            return response;
        }
        /**
         * Get items of a specific post type with WordPress-style query parameters
         * @param postType - Post type to query (e.g., 'product', 'event', etc.)
         * @param params - WordPress query parameters (similar to WP_Query)
         * @returns Promise<WPResponse<WPPost[] | WPPage[]>> - Includes items data and response headers (e.g., x-wp-total, x-wp-totalpages)
         */
        async get_items(postType, params = {}) {
            return this.getPostsByType(postType, params);
        }
        /**
         * Get a single post by ID
         * @param id - Post ID
         * @param params - Additional query parameters
         * @returns Promise<WPResponse<WPPost>>
         */
        async get_post(id, params = {}) {
            const response = await this.client.get(`posts/${id}`, params);
            return response;
        }
        /**
         * Get a single page by ID
         * @param id - Page ID
         * @param params - Additional query parameters
         * @returns Promise<WPResponse<WPPage>>
         */
        async get_page(id, params = {}) {
            const response = await this.client.get(`pages/${id}`, params);
            return response;
        }
        /**
         * Get a single item by ID and post type
         * @param id - Item ID
         * @param postType - Post type (e.g., 'post', 'page', 'product', etc.)
         * @param params - Additional query parameters
         * @returns Promise<WPResponse<WPPost | WPPage>>
         */
        async get_item(id, postType, params = {}) {
            const endpoint = mapPostTypeToEndpoint(postType);
            const response = await this.client.get(`${endpoint}/${id}`, params);
            return response;
        }
        /**
         * Transform advanced query parameters to WordPress REST API format
         * @param params - Advanced query parameters
         * @returns Transformed parameters for the API
         */
        transformQueryParams(params) {
            const apiParams = {};
            // Basic pagination and ordering
            if (params.page)
                apiParams.page = params.page;
            if (params.per_page)
                apiParams.per_page = params.per_page;
            if (params.orderby)
                apiParams.orderby = params.orderby;
            if (params.order)
                apiParams.order = params.order;
            if (params.offset)
                apiParams.offset = params.offset;
            // Post type and status
            if (params.post_type && params.post_type !== 'posts') ;
            if (params.post_status)
                apiParams.status = params.post_status;
            // Search and filtering
            if (params.search)
                apiParams.search = params.search;
            if (params.author) {
                if (Array.isArray(params.author)) {
                    apiParams.author = params.author.join(',');
                }
                else {
                    apiParams.author = params.author;
                }
            }
            if (params.author_exclude) {
                if (Array.isArray(params.author_exclude)) {
                    apiParams.author_exclude = params.author_exclude.join(',');
                }
                else {
                    apiParams.author_exclude = params.author_exclude;
                }
            }
            // Date filtering
            if (params.before)
                apiParams.before = params.before;
            if (params.after)
                apiParams.after = params.after;
            if (params.modified_before)
                apiParams.modified_before = params.modified_before;
            if (params.modified_after)
                apiParams.modified_after = params.modified_after;
            // Include/exclude specific posts
            if (params.include) {
                if (Array.isArray(params.include)) {
                    apiParams.include = params.include.join(',');
                }
                else {
                    apiParams.include = params.include;
                }
            }
            if (params.exclude) {
                if (Array.isArray(params.exclude)) {
                    apiParams.exclude = params.exclude.join(',');
                }
                else {
                    apiParams.exclude = params.exclude;
                }
            }
            // Slug filtering
            if (params.slug) {
                if (Array.isArray(params.slug)) {
                    apiParams.slug = params.slug.join(',');
                }
                else {
                    apiParams.slug = params.slug;
                }
            }
            // Parent filtering (for hierarchical post types)
            if (params.parent) {
                if (Array.isArray(params.parent)) {
                    apiParams.parent = params.parent.join(',');
                }
                else {
                    apiParams.parent = params.parent;
                }
            }
            if (params.parent_exclude) {
                if (Array.isArray(params.parent_exclude)) {
                    apiParams.parent_exclude = params.parent_exclude.join(',');
                }
                else {
                    apiParams.parent_exclude = params.parent_exclude;
                }
            }
            // Taxonomy queries
            if (params.categories) {
                if (Array.isArray(params.categories)) {
                    apiParams.categories = params.categories.join(',');
                }
                else {
                    apiParams.categories = params.categories;
                }
            }
            if (params.categories_exclude) {
                if (Array.isArray(params.categories_exclude)) {
                    apiParams.categories_exclude = params.categories_exclude.join(',');
                }
                else {
                    apiParams.categories_exclude = params.categories_exclude;
                }
            }
            if (params.tags) {
                if (Array.isArray(params.tags)) {
                    apiParams.tags = params.tags.join(',');
                }
                else {
                    apiParams.tags = params.tags;
                }
            }
            if (params.tags_exclude) {
                if (Array.isArray(params.tags_exclude)) {
                    apiParams.tags_exclude = params.tags_exclude.join(',');
                }
                else {
                    apiParams.tags_exclude = params.tags_exclude;
                }
            }
            // Advanced taxonomy query
            if (params.tax_query && params.tax_query.length > 0) {
                // Transform tax_query array to individual taxonomy parameters
                params.tax_query.forEach((taxQuery, index) => {
                    if (taxQuery.taxonomy && taxQuery.terms) {
                        const terms = Array.isArray(taxQuery.terms) ? taxQuery.terms.join(',') : taxQuery.terms;
                        if (taxQuery.operator === 'NOT IN') {
                            apiParams[`${taxQuery.taxonomy}_exclude`] = terms;
                        }
                        else {
                            apiParams[taxQuery.taxonomy] = terms;
                        }
                    }
                });
            }
            // Meta queries - simplified approach
            if (params.meta_key)
                apiParams.meta_key = params.meta_key;
            if (params.meta_value)
                apiParams.meta_value = params.meta_value;
            if (params.meta_compare)
                apiParams.meta_compare = params.meta_compare;
            if (params.meta_type)
                apiParams.meta_type = params.meta_type;
            // Advanced meta query - convert to simple meta_key/meta_value where possible
            if (params.meta_query && params.meta_query.length > 0) {
                const firstMetaQuery = params.meta_query[0];
                if (firstMetaQuery.key && firstMetaQuery.value) {
                    apiParams.meta_key = firstMetaQuery.key;
                    apiParams.meta_value = firstMetaQuery.value;
                    if (firstMetaQuery.compare)
                        apiParams.meta_compare = firstMetaQuery.compare;
                    if (firstMetaQuery.type)
                        apiParams.meta_type = firstMetaQuery.type;
                }
            }
            // Sticky posts
            if (params.sticky !== undefined)
                apiParams.sticky = params.sticky;
            // Context and fields
            if (params.context)
                apiParams.context = params.context;
            if (params._fields)
                apiParams._fields = params._fields.join(',');
            if (params._embed !== undefined)
                apiParams._embed = params._embed.join(',');
            return apiParams;
        }
        /**
         * Helper method to handle different post types
         * @param postType - Post type to query
         * @param params - Query parameters
         * @returns Promise<WPResponse<WPPost[]>>
         */
        async getPostsByType(postType, params = {}) {
            // Map WordPress post type to REST API endpoint
            const endpoint = mapPostTypeToEndpoint(postType);
            if (postType === 'post') {
                return this.get_posts(params);
            }
            else {
                // Try custom post type endpoint first, fallback to posts with type filter
                try {
                    const response = await this.client.get(endpoint, this.transformQueryParams(params));
                    return response;
                }
                catch (error) {
                    // Fallback: add post_type to the posts endpoint
                    const paramsWithType = { ...params, post_type: postType };
                    const response = await this.client.get('posts', this.transformQueryParams(paramsWithType));
                    return response;
                }
            }
        }
    }

    /**
     * WordPress Taxonomies API
     * Simplified to core functions that handle everything through parameters
     */
    class WPTaxonomiesAPI {
        constructor(client) {
            this.client = client;
        }
        /**
         * Get terms with WordPress-style query parameters
         * This is the main function that handles all term queries
         * @param params - WordPress taxonomy query parameters
         * @returns Promise<WPResponse<WPTerm[]>> - Includes terms data and response headers
         */
        async get_terms(params = {}) {
            const endpoint = mapTaxonomyToEndpoint(params.taxonomy || 'category');
            const response = await this.client.get(endpoint, this.transformQueryParams(params));
            return response;
        }
        /**
         * Get taxonomies
         * @param params - Query parameters
         * @returns Promise<WPResponse<WPTaxonomy[]>> - Includes taxonomies data and response headers
         */
        async get_taxonomies(params = {}) {
            const response = await this.client.get('taxonomies', params);
            return response;
        }
        /**
         * Transform taxonomy query parameters to WordPress REST API format
         * @param params - Taxonomy query parameters
         * @returns Transformed parameters for the API
         */
        transformQueryParams(params) {
            const apiParams = {};
            // Basic pagination and ordering
            if (params.page)
                apiParams.page = params.page;
            if (params.per_page)
                apiParams.per_page = params.per_page;
            if (params.orderby)
                apiParams.orderby = params.orderby;
            if (params.order)
                apiParams.order = params.order;
            if (params.offset)
                apiParams.offset = params.offset;
            // Search and filtering
            if (params.search)
                apiParams.search = params.search;
            if (params.name)
                apiParams.name = params.name;
            if (params.slug) {
                if (Array.isArray(params.slug)) {
                    apiParams.slug = params.slug.join(',');
                }
                else {
                    apiParams.slug = params.slug;
                }
            }
            // Include/exclude specific terms
            if (params.include) {
                if (Array.isArray(params.include)) {
                    apiParams.include = params.include.join(',');
                }
                else {
                    apiParams.include = params.include;
                }
            }
            if (params.exclude) {
                if (Array.isArray(params.exclude)) {
                    apiParams.exclude = params.exclude.join(',');
                }
                else {
                    apiParams.exclude = params.exclude;
                }
            }
            // Hierarchical filtering
            if (params.parent !== undefined)
                apiParams.parent = params.parent;
            if (params.child_of)
                apiParams.child_of = params.child_of;
            // Taxonomy-specific parameters
            if (params.hide_empty !== undefined)
                apiParams.hide_empty = params.hide_empty;
            if (params.post)
                apiParams.post = params.post;
            if (params.count !== undefined)
                apiParams.count = params.count;
            if (params.fields)
                apiParams.fields = params.fields;
            if (params.get)
                apiParams.get = params.get;
            // Context and fields
            if (params.context)
                apiParams.context = params.context;
            if (params._fields)
                apiParams._fields = params._fields;
            if (params._embed !== undefined)
                apiParams._embed = params._embed;
            // Remove taxonomy from params as it's used for endpoint
            const { taxonomy, ...restParams } = params;
            // Add any additional custom parameters
            Object.keys(restParams).forEach(key => {
                if (!apiParams.hasOwnProperty(key) && restParams[key] !== undefined) {
                    apiParams[key] = restParams[key];
                }
            });
            return apiParams;
        }
    }

    /**
     * WordPress Meta API
     * Simplified to core functions that handle everything through parameters
     */
    class WPMetaAPI {
        constructor(client) {
            this.client = client;
        }
        /**
         * Get meta data with WordPress-style query parameters
         * This is the main function that handles all meta queries
         * @param params - WordPress meta query parameters
         * @returns Promise<WPResponse<any[]>> - Includes meta data and response headers
         */
        async get_meta(params = {}) {
            const endpoint = this.buildEndpoint(params);
            const response = await this.client.get(endpoint, this.transformQueryParams(params));
            return response;
        }
        /**
         * Create new meta data
         * @param objectType - Object type (post, page, term, user, comment)
         * @param objectId - Object ID
         * @param metaData - Meta data to create
         * @returns Promise<WPResponse<any>> - Includes created meta data and response headers
         */
        async create_meta(objectType, objectId, metaData) {
            const endpoint = `${objectType}s/${objectId}/meta`;
            const response = await this.client.post(endpoint, metaData);
            return response;
        }
        /**
         * Update existing meta data
         * @param objectType - Object type (post, page, term, user, comment)
         * @param objectId - Object ID
         * @param metaId - Meta ID
         * @param metaData - Meta data to update
         * @returns Promise<WPResponse<any>> - Includes updated meta data and response headers
         */
        async update_meta(objectType, objectId, metaId, metaData) {
            const endpoint = `${objectType}s/${objectId}/meta/${metaId}`;
            const response = await this.client.put(endpoint, metaData);
            return response;
        }
        /**
         * Delete meta data
         * @param objectType - Object type (post, page, term, user, comment)
         * @param objectId - Object ID
         * @param metaId - Meta ID
         * @param force - Whether to force delete
         * @returns Promise<WPResponse<any>> - Includes deletion result and response headers
         */
        async delete_meta(objectType, objectId, metaId, force = false) {
            const endpoint = `${objectType}s/${objectId}/meta/${metaId}`;
            const response = await this.client.delete(endpoint, { force });
            return response;
        }
        /**
         * Build the appropriate endpoint based on parameters
         * @param params - Meta query parameters
         * @returns Endpoint string
         */
        buildEndpoint(params) {
            const objectType = params.object_type || 'post';
            const objectId = params.object_id;
            if (objectId) {
                return `${objectType}s/${objectId}/meta`;
            }
            else {
                // Generic meta endpoint (if supported)
                return `${objectType}s/meta`;
            }
        }
        /**
         * Transform meta query parameters to WordPress REST API format
         * @param params - Meta query parameters
         * @returns Transformed parameters for the API
         */
        transformQueryParams(params) {
            const apiParams = {};
            // Basic pagination and ordering
            if (params.page)
                apiParams.page = params.page;
            if (params.per_page)
                apiParams.per_page = params.per_page;
            if (params.orderby)
                apiParams.orderby = params.orderby;
            if (params.order)
                apiParams.order = params.order;
            if (params.offset)
                apiParams.offset = params.offset;
            // Search and filtering
            if (params.search)
                apiParams.search = params.search;
            if (params.key)
                apiParams.key = params.key;
            if (params.value !== undefined)
                apiParams.value = params.value;
            if (params.compare)
                apiParams.compare = params.compare;
            if (params.type)
                apiParams.type = params.type;
            // Include/exclude specific meta
            if (params.include) {
                if (Array.isArray(params.include)) {
                    apiParams.include = params.include.join(',');
                }
                else {
                    apiParams.include = params.include;
                }
            }
            if (params.exclude) {
                if (Array.isArray(params.exclude)) {
                    apiParams.exclude = params.exclude.join(',');
                }
                else {
                    apiParams.exclude = params.exclude;
                }
            }
            // Context and fields
            if (params.context)
                apiParams.context = params.context;
            if (params._fields)
                apiParams._fields = params._fields;
            if (params._embed !== undefined)
                apiParams._embed = params._embed;
            // Remove object-specific params as they're used for endpoint
            const { object_type, object_id, ...restParams } = params;
            // Add any additional custom parameters
            Object.keys(restParams).forEach(key => {
                if (!apiParams.hasOwnProperty(key) && restParams[key] !== undefined) {
                    apiParams[key] = restParams[key];
                }
            });
            return apiParams;
        }
    }

    /**
     * WordPress Menus API
     * Uses dedicated REST endpoints (/wvc/v1/menus) when available, falls back to posts and taxonomies APIs
     */
    class WPMenusAPI {
        constructor(client) {
            this.client = client;
            this.posts = new WPPostsAPI(client);
            this.taxonomies = new WPTaxonomiesAPI(client);
        }
        /**
         * Transform a WordPress term (nav_menu) to WPMenu format
         * @param term - WordPress term from nav_menu taxonomy
         * @returns WPMenu object
         */
        transformTermToMenu(term) {
            return {
                id: term.id,
                name: term.name,
                slug: term.slug,
                description: term.description,
                count: term.count,
                items: [] // Will be populated separately
            };
        }
        /**
         * Transform a WordPress post (nav_menu_item) to WPMenuItem format
         * @param post - WordPress post of type nav_menu_item
         * @returns WPMenuItem object
         */
        transformPostToMenuItem(post) {
            var _a, _b;
            const meta = post.meta || {};
            return {
                id: post.id,
                title: ((_a = post.title) === null || _a === void 0 ? void 0 : _a.rendered) || '',
                url: meta._menu_item_url || post.link || '',
                target: meta._menu_item_target || '',
                description: ((_b = post.content) === null || _b === void 0 ? void 0 : _b.rendered) || '',
                classes: meta._menu_item_classes ? meta._menu_item_classes.split(' ') : [],
                menu_order: post.menu_order || 0,
                parent: parseInt(meta._menu_item_menu_item_parent) || 0,
                object_id: parseInt(meta._menu_item_object_id) || 0,
                object: meta._menu_item_object || '',
                type: meta._menu_item_type || '',
            };
        }
        /**
         * Get menus with WordPress-style query parameters
         * Uses the new /menus REST endpoint when available, falls back to taxonomies API
         * @param params - WordPress query parameters
         * @returns Promise<WPResponse<WPMenu[]>> - Includes menus data and response headers
         */
        async get_menus(params = {}) {
            try {
                // Try the new REST endpoint first with wvc/v1 namespace
                const response = await this.client.getWithNamespace('menus', params, 'wvc/v1');
                const menus = response.data.map((menu) => ({
                    id: menu.id,
                    name: menu.name,
                    slug: menu.slug,
                    description: menu.description || '',
                    count: menu.count || 0,
                    items: menu.items || []
                }));
                return {
                    data: menus,
                    headers: response.headers,
                    status: response.status
                };
            }
            catch (error) {
                // Fallback to taxonomies API for backward compatibility
                const termsResponse = await this.taxonomies.get_terms({
                    ...params,
                    taxonomy: 'nav_menu'
                });
                const menus = termsResponse.data.map(term => this.transformTermToMenu(term));
                return {
                    data: menus,
                    headers: termsResponse.headers,
                    status: termsResponse.status
                };
            }
        }
        /**
         * Get a single menu with its items using the new REST endpoint
         * Corresponds to /menus/(?P<id>\d+) endpoint
         * @param menuId - Menu ID (required)
         * @returns Promise<WPResponse<WPMenu>> - Includes menu data and response headers
         */
        async get_menu(menuId) {
            if (!menuId) {
                throw new Error("menuId is required");
            }
            try {
                // Use the new REST endpoint with wvc/v1 namespace
                const response = await this.client.getWithNamespace(`menus/${menuId}`, {}, 'wvc/v1');
                const menu = response.data;
                const menuData = {
                    id: menu.id,
                    name: menu.name,
                    slug: menu.slug,
                    description: menu.description || '',
                    count: menu.count || 0,
                    items: menu.items || []
                };
                return {
                    data: menuData,
                    headers: response.headers,
                    status: response.status
                };
            }
            catch (error) {
                // Fallback: get menu info using taxonomies API
                const menuTermsResponse = await this.taxonomies.get_terms({
                    taxonomy: 'nav_menu',
                    include: [menuId]
                });
                if (menuTermsResponse.data.length === 0) {
                    throw new Error(`Menu with ID ${menuId} not found`);
                }
                const menu = this.transformTermToMenu(menuTermsResponse.data[0]);
                // Get items using posts API as fallback
                const postsResponse = await this.posts.get_posts({
                    post_type: 'nav_menu_item',
                    meta_query: [{
                            key: '_menu_item_menu_item_parent',
                            value: menuId.toString(),
                            compare: '='
                        }],
                    per_page: 100
                });
                menu.items = postsResponse.data.map(post => this.transformPostToMenuItem(post));
                return {
                    data: menu,
                    headers: postsResponse.headers,
                    status: postsResponse.status
                };
            }
        }
        /**
         * Get menu items for a specific menu using the new REST API structure
         * @param menuId - Menu ID (required)
         * @returns Promise<WPResponse<WPMenuItem[]>> - Includes menu items data and response headers
         */
        async get_menu_items(menuId) {
            if (!menuId) {
                throw new Error("menuId is required");
            }
            try {
                // Get the menu with items using the /menus/{id} endpoint
                // The menu response includes the items array
                const response = await this.client.getWithNamespace(`menus/${menuId}`, {}, 'wvc/v1');
                const menu = response.data;
                // Return the items from the menu response
                const items = (menu.items || []).map((item) => {
                    return {
                        id: item.id,
                        title: item.title || '',
                        url: item.url || '',
                        target: item.target || '',
                        description: item.description || '',
                        classes: item.classes || [],
                        menu_order: item.order || item.menu_order || 0,
                        parent: item.parent || 0,
                        object_id: item.object_id || 0,
                        object: item.object_type || item.object || '',
                        type: item.type || item.object_type || 'custom',
                        resolved_url: item.resolved_url || item.url || '',
                        _menu_item_type: item.type || item.object_type || 'custom',
                        _menu_item_object_id: parseInt(item.object_id) || 0,
                    };
                });
                return {
                    data: items,
                    headers: response.headers,
                    status: response.status
                };
            }
            catch (error) {
                // Fallback to posts API for backward compatibility
                const postsResponse = await this.posts.get_posts({
                    post_type: 'nav_menu_item',
                    meta_query: [{
                            key: '_menu_item_menu_item_parent',
                            value: menuId.toString(),
                            compare: '='
                        }],
                    per_page: 100
                });
                const items = postsResponse.data.map(post => this.transformPostToMenuItem(post));
                return {
                    data: items,
                    headers: postsResponse.headers,
                    status: postsResponse.status
                };
            }
        }
    }

    /**
     * Main WordPress REST API Client
     * Simplified to core functionality with powerful parameter-based queries
     */
    class WordPress {
        constructor(config) {
            this.client = new WPClient(config);
            // Initialize API modules
            this.posts = new WPPostsAPI(this.client);
            this.taxonomies = new WPTaxonomiesAPI(this.client);
            this.meta = new WPMetaAPI(this.client);
            this.menus = new WPMenusAPI(this.client);
        }
        /**
         * Shorthand for posts.get_posts() - WordPress-style query function
         * @param params - WordPress query parameters (similar to WP_Query)
         * @param flushCache - Whether to flush cache for this request
         * @returns Promise<WPResponse<WPPost[]>>
         */
        async get_posts(params = {}, flushCache = false) {
            if (flushCache) {
                // If flushCache is true, we need to call the client directly with flushCache
                const response = await this.client.get('posts', this.posts['transformQueryParams'](params), flushCache);
                return response;
            }
            return this.posts.get_posts(params);
        }
        /**
         * Shorthand for posts.get_pages() - Get pages with WordPress-style query parameters
         * @param params - WordPress query parameters (similar to WP_Query)
         * @param flushCache - Whether to flush cache for this request
         * @returns Promise<WPResponse<WPPage[]>>
         */
        async get_pages(params = {}, flushCache = false) {
            if (flushCache) {
                const response = await this.client.get('pages', this.posts['transformQueryParams'](params), flushCache);
                return response;
            }
            return this.posts.get_pages(params);
        }
        /**
         * Shorthand for posts.get_items() - Get items of a specific post type
         * @param postType - Post type to query (e.g., 'product', 'event', etc.)
         * @param params - WordPress query parameters (similar to WP_Query)
         * @param flushCache - Whether to flush cache for this request
         * @returns Promise<WPResponse<WPPost[] | WPPage[]>>
         */
        async get_items(postType, params = {}, flushCache = false) {
            if (flushCache) {
                const endpoint = mapPostTypeToEndpoint(postType);
                const response = await this.client.get(endpoint, this.posts['transformQueryParams'](params), flushCache);
                return response;
            }
            return this.posts.get_items(postType, params);
        }
        /**
         * Shorthand for posts.get_post() - Get a single post by ID
         * @param id - Post ID
         * @param params - Additional query parameters
         * @param flushCache - Whether to flush cache for this request
         * @returns Promise<WPPost>
         */
        async get_post(id, params = {}, flushCache = false) {
            if (flushCache) {
                const response = await this.client.get(`posts/${id}`, params, flushCache);
                return response.data;
            }
            return this.posts.get_post(id, params);
        }
        /**
         * Shorthand for posts.get_page() - Get a single page by ID
         * @param id - Page ID
         * @param params - Additional query parameters
         * @param flushCache - Whether to flush cache for this request
         * @returns Promise<WPPage>
         */
        async get_page(id, params = {}, flushCache = false) {
            if (flushCache) {
                const response = await this.client.get(`pages/${id}`, params, flushCache);
                return response.data;
            }
            return this.posts.get_page(id, params);
        }
        /**
         * Shorthand for posts.get_item() - Get a single item by ID and post type
         * @param id - Item ID
         * @param postType - Post type (e.g., 'post', 'page', 'product', etc.)
         * @param params - Additional query parameters
         * @param flushCache - Whether to flush cache for this request
         * @returns Promise<WPPost | WPPage>
         */
        async get_item(id, postType, params = {}, flushCache = false) {
            if (flushCache) {
                const endpoint = mapPostTypeToEndpoint(postType);
                const response = await this.client.get(`${endpoint}/${id}`, params, flushCache);
                return response.data;
            }
            return this.posts.get_item(id, postType, params);
        }
        /**
         * Shorthand for taxonomies.get_terms() - WordPress-style terms query
         * @param params - Taxonomy query parameters
         * @param flushCache - Whether to flush cache for this request
         * @returns Promise<WPResponse<WPTerm[]>>
         */
        async get_terms(params = {}, flushCache = false) {
            if (flushCache) {
                const endpoint = mapTaxonomyToEndpoint(params.taxonomy || 'category');
                const response = await this.client.get(endpoint, this.taxonomies['transformQueryParams'](params), flushCache);
                return response;
            }
            return this.taxonomies.get_terms(params);
        }
        /**
         * Shorthand for taxonomies.get_taxonomies() - WordPress-style taxonomies query
         * @param params - Query parameters
         * @param flushCache - Whether to flush cache for this request
         * @returns Promise<WPResponse<WPTaxonomy[]>>
         */
        async get_taxonomies(params = {}, flushCache = false) {
            if (flushCache) {
                const response = await this.client.get('taxonomies', params, flushCache);
                return response;
            }
            return this.taxonomies.get_taxonomies(params);
        }
        /**
         * Shorthand for meta.get_meta() - WordPress-style meta query
         * @param params - Meta query parameters
         * @param flushCache - Whether to flush cache for this request
         * @returns Promise<WPResponse<any[]>>
         */
        async get_meta(params = {}, flushCache = false) {
            if (flushCache) {
                const endpoint = this.meta['buildEndpoint'](params);
                const response = await this.client.get(endpoint, this.meta['transformQueryParams'](params), flushCache);
                return response;
            }
            return this.meta.get_meta(params);
        }
        /**
         * Shorthand for menus.get_menus() - WordPress-style menus query
         * @param params - Query parameters
         * @param flushCache - Whether to flush cache for this request
         * @returns Promise<WPResponse<WPMenu[]>>
         */
        async get_menus(params = {}, flushCache = false) {
            if (flushCache) {
                const response = await this.client.getWithNamespace('menus', params, 'wvc/v1', flushCache);
                const menus = response.data.map((menu) => ({
                    id: menu.id,
                    name: menu.name,
                    slug: menu.slug,
                    description: menu.description || '',
                    count: menu.count || 0,
                    items: menu.items || []
                }));
                return {
                    data: menus,
                    headers: response.headers,
                    status: response.status
                };
            }
            return this.menus.get_menus(params);
        }
        /**
         * Shorthand for menus.get_menu() - Get single menu with items
         * @param menuId - Menu ID (required)
         * @param flushCache - Whether to flush cache for this request
         * @returns Promise<any>
         */
        async get_menu(menuId, flushCache = false) {
            if (flushCache) {
                const response = await this.client.getWithNamespace(`menus/${menuId}`, {}, 'wvc/v1', flushCache);
                const menu = response.data;
                return {
                    id: menu.id,
                    name: menu.name,
                    slug: menu.slug,
                    description: menu.description || '',
                    count: menu.count || 0,
                    items: menu.items || []
                };
            }
            return this.menus.get_menu(menuId);
        }
        /**
         * Shorthand for menus.get_menu_items() - WordPress-style menu items query
         * @param menuId - Menu ID (required)
         * @param params - Query parameters
         * @param flushCache - Whether to flush cache for this request
         * @returns Promise<WPResponse<WPMenuItem[]>>
         */
        async get_menu_items(menuId, params = {}, flushCache = false) {
            if (flushCache) {
                const response = await this.client.getWithNamespace(`menus/${menuId}`, {}, 'wvc/v1', flushCache);
                const menu = response.data;
                // Return the items from the menu response
                const items = (menu.items || []).map((item) => {
                    return {
                        id: item.id,
                        title: item.title || '',
                        url: item.url || '',
                        target: item.target || '',
                        description: item.description || '',
                        classes: item.classes || [],
                        menu_order: item.order || item.menu_order || 0,
                        parent: item.parent || 0,
                        object_id: item.object_id || 0,
                        object: item.object_type || item.object || '',
                        type: item.type || item.object_type || 'custom',
                        resolved_url: item.resolved_url || item.url || '',
                        _menu_item_type: item.type || item.object_type || 'custom',
                        _menu_item_object_id: parseInt(item.object_id) || 0,
                    };
                });
                return {
                    data: items,
                    headers: response.headers,
                    status: response.status
                };
            }
            return this.menus.get_menu_items(menuId);
        }
        // =============================================================================
        // UTILITY METHODS FOR WPResponse HANDLING
        // =============================================================================
        /**
         * Generic method to extract body/data from any WPResponse promise
         * Useful for backward compatibility when you want just the data
         * @param wpResponsePromise - Promise that resolves to WPResponse<T>
         * @returns Promise<T> - Just the data portion
         */
        async get_wp_response_body(wpResponsePromise) {
            const response = await wpResponsePromise;
            return response.data;
        }
        /**
         * Generic method to extract headers from any WPResponse promise
         * Useful for accessing pagination headers or other response metadata
         * @param wpResponsePromise - Promise that resolves to WPResponse<T>
         * @returns Promise<Record<string, string>> - Just the headers portion
         */
        async get_wp_response_headers(wpResponsePromise) {
            const response = await wpResponsePromise;
            return response.headers;
        }
        // =============================================================================
        // CACHE MANAGEMENT METHODS
        // =============================================================================
        /**
         * Get cache statistics
         * @returns Cache statistics object
         */
        getCacheStats() {
            return this.client.getCacheStats();
        }
        /**
         * Clear all cache entries
         */
        clearCache() {
            this.client.clearCache();
        }
        /**
         * Flush expired cache entries
         */
        flushCache() {
            this.client.flushCache();
        }
        /**
         * Clear cache entries by method name pattern
         * @param methodPattern - Method name pattern (regex string)
         * @returns Number of entries cleared
         */
        clearCacheByMethod(methodPattern) {
            return this.client.clearCacheByMethod(methodPattern);
        }
        /**
         * Flush cache for a specific method and arguments
         * @param methodName - Method name
         * @param args - Method arguments
         */
        flushCacheForMethod(methodName, args = []) {
            this.client.flushCacheForMethod(methodName, args);
        }
        /**
         * Enable caching
         */
        enableCache() {
            this.client.enableCache();
        }
        /**
         * Disable caching
         */
        disableCache() {
            this.client.disableCache();
        }
        /**
         * Check if cache is enabled
         * @returns True if cache is enabled
         */
        isCacheEnabled() {
            return this.client.isCacheEnabled();
        }
    }

    /**
     * WordPress REST API Cache
     *
     * Provides intelligent caching for WordPress REST API requests with:
     * - Method-based cache keys (method name + stringified arguments)
     * - TTL-based expiration
     * - LRU eviction when max entries reached
     * - Configurable cache behavior
     * - Memory management
     */
    class WPCache {
        constructor(config = {}) {
            this.cache = new Map();
            this.accessOrder = new Map(); // For LRU tracking
            this.accessCounter = 0;
            this.config = {
                enabled: true,
                ttl: 24 * 60 * 60 * 1000, // 24 hours default
                maxEntries: 1000,
                keyPrefix: "wp_cache_",
                ...config
            };
            this.stats = {
                hits: 0,
                misses: 0,
                entries: 0,
                maxEntries: this.config.maxEntries,
                hitRate: 0,
                memoryUsage: 0
            };
        }
        /**
         * Generate a cache key based on method name and arguments
         * @param methodName - Method name (e.g., "get_posts", "get_menu")
         * @param args - Method arguments (will be stringified)
         * @returns Cache key string
         */
        generateCacheKey(methodName, args = []) {
            // Sort arguments to ensure consistent cache keys regardless of order
            const sortedArgs = this.sortArguments(args);
            // Create a deterministic string representation of sorted arguments
            const argsString = JSON.stringify(sortedArgs);
            const baseKey = `${methodName}:${argsString}`;
            // Create a hash-like key (simple but effective)
            let hash = 0;
            for (let i = 0; i < baseKey.length; i++) {
                const char = baseKey.charCodeAt(i);
                hash = ((hash << 5) - hash) + char;
                hash = hash & hash; // Convert to 32-bit integer
            }
            return `${this.config.keyPrefix}${Math.abs(hash)}_${methodName.replace(/[^a-zA-Z0-9]/g, "_")}`;
        }
        /**
         * Sort arguments to ensure consistent cache keys
         * @param args - Method arguments
         * @returns Sorted arguments
         */
        sortArguments(args) {
            return args.map(arg => {
                if (typeof arg === 'object' && arg !== null && !Array.isArray(arg)) {
                    // Sort object keys alphabetically
                    const sortedObj = {};
                    Object.keys(arg).sort().forEach(key => {
                        sortedObj[key] = arg[key];
                    });
                    return sortedObj;
                }
                return arg;
            });
        }
        /**
         * Check if a cache entry is expired
         * @param entry - Cache entry
         * @returns True if expired
         */
        isExpired(entry) {
            const now = Date.now();
            return (now - entry.timestamp) > entry.ttl;
        }
        /**
         * Update access order for LRU eviction
         * @param key - Cache key
         */
        updateAccessOrder(key) {
            this.accessOrder.set(key, ++this.accessCounter);
        }
        /**
         * Find least recently used key
         * @returns LRU key or null
         */
        findLRUKey() {
            if (this.accessOrder.size === 0)
                return null;
            let lruKey = "";
            let lruCount = Number.MAX_SAFE_INTEGER;
            for (const [key, count] of this.accessOrder) {
                if (count < lruCount) {
                    lruCount = count;
                    lruKey = key;
                }
            }
            return lruKey || null;
        }
        /**
         * Evict entries if cache is full
         */
        evictIfNeeded() {
            while (this.cache.size >= this.config.maxEntries) {
                const lruKey = this.findLRUKey();
                if (lruKey) {
                    this.cache.delete(lruKey);
                    this.accessOrder.delete(lruKey);
                }
                else {
                    break;
                }
            }
        }
        /**
         * Clean expired entries from cache
         */
        cleanExpired() {
            const expiredKeys = [];
            for (const [key, entry] of this.cache) {
                if (this.isExpired(entry)) {
                    expiredKeys.push(key);
                }
            }
            expiredKeys.forEach(key => {
                this.cache.delete(key);
                this.accessOrder.delete(key);
            });
        }
        /**
         * Update cache statistics
         */
        updateStats() {
            this.stats.entries = this.cache.size;
            this.stats.hitRate = this.stats.hits + this.stats.misses > 0
                ? (this.stats.hits / (this.stats.hits + this.stats.misses)) * 100
                : 0;
            // Rough memory usage calculation
            this.stats.memoryUsage = this.cache.size * 1024; // Approximate 1KB per entry
        }
        /**
         * Get cached data by method name and arguments
         * @param methodName - Method name
         * @param args - Method arguments
         * @returns Cached data or null if not found/expired
         */
        get(methodName, args = []) {
            if (!this.config.enabled) {
                return null;
            }
            const key = this.generateCacheKey(methodName, args);
            const entry = this.cache.get(key);
            if (!entry) {
                this.stats.misses++;
                this.updateStats();
                return null;
            }
            if (this.isExpired(entry)) {
                this.cache.delete(key);
                this.accessOrder.delete(key);
                this.stats.misses++;
                this.updateStats();
                return null;
            }
            this.updateAccessOrder(key);
            this.stats.hits++;
            this.updateStats();
            return entry.data;
        }
        /**
         * Set cached data by method name and arguments
         * @param methodName - Method name
         * @param args - Method arguments
         * @param data - Data to cache
         * @param customTtl - Custom TTL for this entry
         */
        set(methodName, args = [], data, customTtl) {
            if (!this.config.enabled) {
                return;
            }
            // Clean expired entries periodically
            if (this.cache.size % 100 === 0) {
                this.cleanExpired();
            }
            this.evictIfNeeded();
            const key = this.generateCacheKey(methodName, args);
            const ttl = customTtl || this.config.ttl;
            const entry = {
                data,
                timestamp: Date.now(),
                ttl,
                key
            };
            this.cache.set(key, entry);
            this.updateAccessOrder(key);
            this.updateStats();
        }
        /**
         * Check if data exists in cache (and is not expired)
         * @param methodName - Method name
         * @param args - Method arguments
         * @returns True if cached data exists
         */
        has(methodName, args = []) {
            return this.get(methodName, args) !== null;
        }
        /**
         * Remove specific cache entry
         * @param methodName - Method name
         * @param args - Method arguments
         */
        delete(methodName, args = []) {
            if (!this.config.enabled) {
                return false;
            }
            const key = this.generateCacheKey(methodName, args);
            const deleted = this.cache.delete(key);
            if (deleted) {
                this.accessOrder.delete(key);
                this.updateStats();
            }
            return deleted;
        }
        /**
         * Flush cache for a specific method and arguments
         * @param methodName - Method name
         * @param args - Method arguments
         */
        flush(methodName, args = []) {
            this.delete(methodName, args);
        }
        /**
         * Clear all cache entries
         */
        clear() {
            this.cache.clear();
            this.accessOrder.clear();
            this.stats.hits = 0;
            this.stats.misses = 0;
            this.updateStats();
        }
        /**
         * Flush expired entries
         */
        flushExpired() {
            this.cleanExpired();
            this.updateStats();
        }
        /**
         * Clear cache entries by method name pattern
         * @param methodPattern - Method name pattern (regex string or exact match)
         */
        clearByMethod(methodPattern) {
            if (!this.config.enabled) {
                return 0;
            }
            const regex = new RegExp(methodPattern);
            const keysToDelete = [];
            for (const [key, entry] of this.cache) {
                // Extract method name from key
                const methodName = key.split('_').slice(-1)[0]; // Get last part after underscore
                if (regex.test(methodName)) {
                    keysToDelete.push(key);
                }
            }
            keysToDelete.forEach(key => {
                this.cache.delete(key);
                this.accessOrder.delete(key);
            });
            this.updateStats();
            return keysToDelete.length;
        }
        /**
         * Get cache statistics
         * @returns Cache statistics
         */
        getStats() {
            this.updateStats();
            return { ...this.stats };
        }
        /**
         * Get cache configuration
         * @returns Cache configuration
         */
        getConfig() {
            return { ...this.config };
        }
        /**
         * Update cache configuration
         * @param newConfig - New configuration
         */
        updateConfig(newConfig) {
            this.config = { ...this.config, ...newConfig };
            this.stats.maxEntries = this.config.maxEntries;
            // If cache is disabled, clear everything
            if (!this.config.enabled) {
                this.clear();
            }
            // If max entries reduced, evict excess entries
            if (this.cache.size > this.config.maxEntries) {
                this.evictIfNeeded();
            }
        }
        /**
         * Get all cache entries (for debugging)
         * @returns Array of cache entries
         */
        getEntries() {
            return Array.from(this.cache.values());
        }
        /**
         * Get cache entries by method name
         * @param methodName - Method name to filter by
         * @returns Array of matching cache entries
         */
        getEntriesByMethod(methodName) {
            return Array.from(this.cache.values()).filter(entry => entry.key.includes(methodName));
        }
        /**
         * Enable cache
         */
        enable() {
            this.config.enabled = true;
        }
        /**
         * Disable cache and clear all entries
         */
        disable() {
            this.config.enabled = false;
            this.clear();
        }
        /**
         * Check if cache is enabled
         * @returns True if cache is enabled
         */
        isEnabled() {
            return this.config.enabled;
        }
    }

    /**
     * Generic JSON Schemas for WordPress REST API
     *
     * These schemas provide flexible, extensible definitions for:
     * - Custom post types (full and simplified versions)
     * - Custom taxonomy terms (full and simplified versions)
     * - Custom taxonomies (full and simplified versions)
     *
     * Full schemas handle complete WordPress REST API responses
     * Simplified schemas focus on content creation and user input
     */
    /**
     * JSON Schema definitions for validation
     */
    // Simplified JSON Schemas (for content creation)
    const WPSimplePostJSONSchema = {
        type: "object",
        properties: {
            title: { type: "string" },
            content: { type: "string" },
            excerpt: { type: "string" },
            slug: { type: "string" },
            status: {
                type: "string",
                enum: ["publish", "draft", "private", "pending"]
            },
            type: { type: "string" },
            parent: { type: "number" },
            menu_order: { type: "number" },
            taxonomies: {
                type: "object",
                additionalProperties: {
                    type: "array",
                    items: {
                        oneOf: [
                            { type: "number" },
                            { type: "string" }
                        ]
                    }
                }
            },
            categories: {
                type: "array",
                items: {
                    oneOf: [
                        { type: "number" },
                        { type: "string" }
                    ]
                }
            },
            tags: {
                type: "array",
                items: {
                    oneOf: [
                        { type: "number" },
                        { type: "string" }
                    ]
                }
            },
            meta: { type: "object" },
            acf: { type: "object" }
        },
        required: ["title", "type"],
        additionalProperties: true
    };
    const WPSimpleTermJSONSchema = {
        type: "object",
        properties: {
            name: { type: "string" },
            slug: { type: "string" },
            description: { type: "string" },
            taxonomy: { type: "string" },
            parent: { type: "number" },
            meta: { type: "object" },
            acf: { type: "object" },
            color: { type: "string" },
            icon: { type: "string" },
            image: {
                oneOf: [
                    { type: "string" },
                    { type: "object" }
                ]
            }
        },
        required: ["name", "taxonomy"],
        additionalProperties: true
    };
    const WPSimpleTaxonomyJSONSchema = {
        type: "object",
        properties: {
            name: { type: "string" },
            slug: { type: "string" },
            description: { type: "string" },
            types: { type: "array", items: { type: "string" } },
            hierarchical: { type: "boolean" },
            labels: {
                type: "object",
                properties: {
                    name: { type: "string" },
                    singular_name: { type: "string" },
                    add_new_item: { type: "string" },
                    edit_item: { type: "string" }
                },
                required: ["name", "singular_name"],
                additionalProperties: { type: "string" }
            },
            public: { type: "boolean" },
            show_ui: { type: "boolean" },
            show_in_rest: { type: "boolean" },
            rewrite: {
                oneOf: [
                    { type: "object" },
                    { type: "boolean" }
                ]
            },
            custom_settings: { type: "object" }
        },
        required: ["name", "slug", "types", "labels"],
        additionalProperties: true
    };
    // Full JSON Schemas (for complete API responses)
    const WPGenericPostJSONSchema = {
        type: "object",
        properties: {
            id: { type: "number" },
            date: { type: "string", format: "date-time" },
            date_gmt: { type: "string", format: "date-time" },
            modified: { type: "string", format: "date-time" },
            modified_gmt: { type: "string", format: "date-time" },
            slug: { type: "string" },
            status: { type: "string" },
            type: { type: "string" },
            link: { type: "string", format: "uri" },
            title: {
                type: "object",
                properties: {
                    rendered: { type: "string" },
                    raw: { type: "string" },
                    protected: { type: "boolean" }
                },
                required: ["rendered"]
            },
            content: {
                type: "object",
                properties: {
                    rendered: { type: "string" },
                    raw: { type: "string" },
                    protected: { type: "boolean" },
                    block_version: { type: "number" }
                },
                required: ["rendered", "protected"]
            },
            excerpt: {
                type: "object",
                properties: {
                    rendered: { type: "string" },
                    raw: { type: "string" },
                    protected: { type: "boolean" }
                },
                required: ["rendered", "protected"]
            },
            guid: {
                type: "object",
                properties: {
                    rendered: { type: "string" },
                    raw: { type: "string" }
                },
                required: ["rendered"]
            },
            author: { type: "number" },
            featured_media: { type: "number" },
            comment_status: { type: "string", enum: ["open", "closed"] },
            ping_status: { type: "string", enum: ["open", "closed"] },
            sticky: { type: "boolean" },
            template: { type: "string" },
            format: { type: "string" },
            parent: { type: "number" },
            menu_order: { type: "number" },
            taxonomies: {
                type: "object",
                additionalProperties: {
                    oneOf: [
                        { type: "array", items: { type: "number" } },
                        { type: "array", items: { type: "string" } },
                        { type: "array", items: { type: "object" } }
                    ]
                }
            },
            categories: { type: "array", items: { type: "number" } },
            tags: { type: "array", items: { type: "number" } },
            meta: { type: "object" },
            acf: { type: "object" },
            custom_fields: { type: "object" },
            _links: { type: "object" },
            _embedded: { type: "object" }
        },
        required: [
            "id", "date", "date_gmt", "modified", "modified_gmt",
            "slug", "status", "type", "link", "title", "content",
            "excerpt", "guid", "author", "featured_media",
            "comment_status", "ping_status"
        ],
        additionalProperties: true
    };
    const WPGenericTermJSONSchema = {
        type: "object",
        properties: {
            id: { type: "number" },
            count: { type: "number" },
            description: { type: "string" },
            link: { type: "string", format: "uri" },
            name: { type: "string" },
            slug: { type: "string" },
            taxonomy: { type: "string" },
            parent: { type: "number" },
            meta: { type: "object" },
            acf: { type: "object" },
            custom_fields: { type: "object" },
            color: { type: "string" },
            icon: { type: "string" },
            image: {
                oneOf: [
                    { type: "string" },
                    { type: "object" }
                ]
            },
            children: {
                type: "array",
                items: { $ref: "#/definitions/WPGenericTermSchema" }
            },
            ancestors: { type: "array", items: { type: "number" } },
            _links: { type: "object" }
        },
        required: [
            "id", "count", "description", "link", "name", "slug", "taxonomy", "parent"
        ],
        additionalProperties: true
    };
    const WPGenericTaxonomyJSONSchema = {
        type: "object",
        properties: {
            name: { type: "string" },
            slug: { type: "string" },
            description: { type: "string" },
            types: { type: "array", items: { type: "string" } },
            hierarchical: { type: "boolean" },
            rest_base: { type: "string" },
            rest_controller_class: { type: "string" },
            rest_namespace: { type: "string" },
            labels: {
                type: "object",
                properties: {
                    name: { type: "string" },
                    singular_name: { type: "string" }
                },
                required: ["name", "singular_name"],
                additionalProperties: { type: "string" }
            },
            capabilities: { type: "object" },
            public: { type: "boolean" },
            publicly_queryable: { type: "boolean" },
            show_ui: { type: "boolean" },
            show_in_menu: { type: "boolean" },
            show_in_nav_menus: { type: "boolean" },
            show_in_rest: { type: "boolean" },
            show_tagcloud: { type: "boolean" },
            show_in_quick_edit: { type: "boolean" },
            show_admin_column: { type: "boolean" },
            query_var: {
                oneOf: [
                    { type: "string" },
                    { type: "boolean" }
                ]
            },
            rewrite: {
                oneOf: [
                    { type: "object" },
                    { type: "boolean" }
                ]
            },
            default_term: { type: "object" },
            sort: { type: "boolean" },
            args: { type: "object" },
            meta_box_cb: {
                oneOf: [
                    { type: "string" },
                    { type: "boolean" }
                ]
            },
            meta_box_sanitize_cb: { type: "string" },
            custom_settings: { type: "object" },
            _links: { type: "object" }
        },
        required: [
            "name", "slug", "description", "types", "hierarchical",
            "rest_base", "rest_controller_class", "labels"
        ],
        additionalProperties: true
    };
    /**
     * Schema validation utilities
     */
    class WPSchemaValidator {
        /**
         * Validate a simple post object against the simplified schema
         * @param post - Post object to validate
         * @returns Validation result
         */
        static validateSimplePost(post) {
            const errors = [];
            const warnings = [];
            // Check required fields
            if (!post.title || typeof post.title !== "string") {
                errors.push("Field 'title' is required and must be a string");
            }
            if (!post.type || typeof post.type !== "string") {
                errors.push("Field 'type' is required and must be a string");
            }
            // Check optional field types
            if (post.content !== undefined && typeof post.content !== "string") {
                errors.push("Field 'content' must be a string");
            }
            if (post.excerpt !== undefined && typeof post.excerpt !== "string") {
                errors.push("Field 'excerpt' must be a string");
            }
            if (post.slug !== undefined && typeof post.slug !== "string") {
                errors.push("Field 'slug' must be a string");
            }
            if (post.status !== undefined && !["publish", "draft", "private", "pending"].includes(post.status)) {
                warnings.push("Field 'status' should be one of: publish, draft, private, pending");
            }
            return {
                isValid: errors.length === 0,
                errors,
                warnings
            };
        }
        /**
         * Validate a simple term object against the simplified schema
         * @param term - Term object to validate
         * @returns Validation result
         */
        static validateSimpleTerm(term) {
            const errors = [];
            const warnings = [];
            // Check required fields
            if (!term.name || typeof term.name !== "string") {
                errors.push("Field 'name' is required and must be a string");
            }
            if (!term.taxonomy || typeof term.taxonomy !== "string") {
                errors.push("Field 'taxonomy' is required and must be a string");
            }
            // Check optional field types
            if (term.slug !== undefined && typeof term.slug !== "string") {
                errors.push("Field 'slug' must be a string");
            }
            if (term.description !== undefined && typeof term.description !== "string") {
                errors.push("Field 'description' must be a string");
            }
            if (term.parent !== undefined && typeof term.parent !== "number") {
                errors.push("Field 'parent' must be a number");
            }
            return {
                isValid: errors.length === 0,
                errors,
                warnings
            };
        }
        /**
         * Validate a simple taxonomy object against the simplified schema
         * @param taxonomy - Taxonomy object to validate
         * @returns Validation result
         */
        static validateSimpleTaxonomy(taxonomy) {
            const errors = [];
            const warnings = [];
            // Check required fields
            if (!taxonomy.name || typeof taxonomy.name !== "string") {
                errors.push("Field 'name' is required and must be a string");
            }
            if (!taxonomy.slug || typeof taxonomy.slug !== "string") {
                errors.push("Field 'slug' is required and must be a string");
            }
            if (!taxonomy.types || !Array.isArray(taxonomy.types)) {
                errors.push("Field 'types' is required and must be an array");
            }
            if (!taxonomy.labels || typeof taxonomy.labels !== "object") {
                errors.push("Field 'labels' is required and must be an object");
            }
            else {
                if (!taxonomy.labels.name) {
                    errors.push("Field 'labels.name' is required");
                }
                if (!taxonomy.labels.singular_name) {
                    errors.push("Field 'labels.singular_name' is required");
                }
            }
            return {
                isValid: errors.length === 0,
                errors,
                warnings
            };
        }
        /**
         * Validate a post object against the generic post schema
         * @param post - Post object to validate
         * @returns Validation result
         */
        static validatePost(post) {
            const errors = [];
            const warnings = [];
            // Check required fields
            const requiredFields = [
                "id", "date", "date_gmt", "modified", "modified_gmt",
                "slug", "status", "type", "link", "title", "content",
                "excerpt", "guid", "author", "featured_media",
                "comment_status", "ping_status"
            ];
            for (const field of requiredFields) {
                if (post[field] === undefined || post[field] === null) {
                    errors.push(`Missing required field: ${field}`);
                }
            }
            // Check field types
            if (post.id !== undefined && typeof post.id !== "number") {
                errors.push("Field 'id' must be a number");
            }
            if (post.title && typeof post.title !== "object") {
                errors.push("Field 'title' must be an object");
            }
            if (post.title && !post.title.rendered) {
                errors.push("Field 'title.rendered' is required");
            }
            // Check for common issues
            if (post.meta && typeof post.meta !== "object") {
                warnings.push("Field 'meta' should be an object");
            }
            if (post.acf && typeof post.acf !== "object") {
                warnings.push("Field 'acf' should be an object");
            }
            return {
                isValid: errors.length === 0,
                errors,
                warnings
            };
        }
        /**
         * Validate a term object against the generic term schema
         * @param term - Term object to validate
         * @returns Validation result
         */
        static validateTerm(term) {
            const errors = [];
            const warnings = [];
            // Check required fields
            const requiredFields = [
                "id", "count", "description", "link", "name", "slug", "taxonomy", "parent"
            ];
            for (const field of requiredFields) {
                if (term[field] === undefined || term[field] === null) {
                    errors.push(`Missing required field: ${field}`);
                }
            }
            // Check field types
            if (term.id !== undefined && typeof term.id !== "number") {
                errors.push("Field 'id' must be a number");
            }
            if (term.count !== undefined && typeof term.count !== "number") {
                errors.push("Field 'count' must be a number");
            }
            if (term.parent !== undefined && typeof term.parent !== "number") {
                errors.push("Field 'parent' must be a number");
            }
            // Check for common issues
            if (term.meta && typeof term.meta !== "object") {
                warnings.push("Field 'meta' should be an object");
            }
            return {
                isValid: errors.length === 0,
                errors,
                warnings
            };
        }
        /**
         * Validate a taxonomy object against the generic taxonomy schema
         * @param taxonomy - Taxonomy object to validate
         * @returns Validation result
         */
        static validateTaxonomy(taxonomy) {
            const errors = [];
            const warnings = [];
            // Check required fields
            const requiredFields = [
                "name", "slug", "description", "types", "hierarchical",
                "rest_base", "rest_controller_class", "labels"
            ];
            for (const field of requiredFields) {
                if (taxonomy[field] === undefined || taxonomy[field] === null) {
                    errors.push(`Missing required field: ${field}`);
                }
            }
            // Check field types
            if (taxonomy.hierarchical !== undefined && typeof taxonomy.hierarchical !== "boolean") {
                errors.push("Field 'hierarchical' must be a boolean");
            }
            if (taxonomy.types && !Array.isArray(taxonomy.types)) {
                errors.push("Field 'types' must be an array");
            }
            if (taxonomy.labels && typeof taxonomy.labels !== "object") {
                errors.push("Field 'labels' must be an object");
            }
            if (taxonomy.labels && !taxonomy.labels.name) {
                errors.push("Field 'labels.name' is required");
            }
            if (taxonomy.labels && !taxonomy.labels.singular_name) {
                errors.push("Field 'labels.singular_name' is required");
            }
            return {
                isValid: errors.length === 0,
                errors,
                warnings
            };
        }
        /**
         * Convert simple post to WordPress API format
         * @param simplePost - Simple post object
         * @returns Post in WordPress API format
         */
        static convertSimplePostToWordPress(simplePost) {
            return {
                title: {
                    rendered: simplePost.title,
                    raw: simplePost.title,
                    protected: false
                },
                content: {
                    rendered: simplePost.content || "",
                    raw: simplePost.content || "",
                    protected: false
                },
                excerpt: {
                    rendered: simplePost.excerpt || "",
                    raw: simplePost.excerpt || "",
                    protected: false
                },
                slug: simplePost.slug || "",
                status: simplePost.status || "draft",
                type: simplePost.type,
                parent: simplePost.parent || 0,
                menu_order: simplePost.menu_order || 0,
                meta: simplePost.meta || {},
                acf: simplePost.acf || {},
                taxonomies: simplePost.taxonomies || {},
                categories: Array.isArray(simplePost.categories)
                    ? simplePost.categories.map(cat => typeof cat === 'string' ? parseInt(cat, 10) : cat).filter(id => !isNaN(id))
                    : [],
                tags: Array.isArray(simplePost.tags)
                    ? simplePost.tags.map(tag => typeof tag === 'string' ? parseInt(tag, 10) : tag).filter(id => !isNaN(id))
                    : [],
                // Default WordPress API fields
                author: 1,
                featured_media: 0,
                comment_status: "open",
                ping_status: "open"
            };
        }
        /**
         * Convert WordPress API post to simple format
         * @param wordpressPost - WordPress API post object
         * @returns Simple post object
         */
        static convertWordPressPostToSimple(wordpressPost) {
            // Convert taxonomies to simple format (extract IDs if they're objects)
            const simpleTaxonomies = {};
            if (wordpressPost.taxonomies) {
                for (const [taxonomy, terms] of Object.entries(wordpressPost.taxonomies)) {
                    if (Array.isArray(terms)) {
                        if (terms.length > 0 && typeof terms[0] === 'object') {
                            // Convert objects to IDs
                            simpleTaxonomies[taxonomy] = terms.map((term) => term.id || term.slug).filter(Boolean);
                        }
                        else {
                            // Already in simple format
                            simpleTaxonomies[taxonomy] = terms;
                        }
                    }
                }
            }
            return {
                title: wordpressPost.title.rendered,
                content: wordpressPost.content.rendered,
                excerpt: wordpressPost.excerpt.rendered,
                slug: wordpressPost.slug,
                status: wordpressPost.status,
                type: wordpressPost.type,
                parent: wordpressPost.parent,
                menu_order: wordpressPost.menu_order,
                meta: wordpressPost.meta,
                acf: wordpressPost.acf,
                taxonomies: simpleTaxonomies,
                ...(wordpressPost.categories && { categories: wordpressPost.categories }),
                ...(wordpressPost.tags && { tags: wordpressPost.tags })
            };
        }
        /**
         * Normalize a post object to match the generic schema
         * @param post - Post object to normalize
         * @returns Normalized post object
         */
        static normalizePost(post) {
            var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
            const normalized = {
                id: post.id || 0,
                date: post.date || "",
                date_gmt: post.date_gmt || "",
                modified: post.modified || "",
                modified_gmt: post.modified_gmt || "",
                slug: post.slug || "",
                status: post.status || "publish",
                type: post.type || "post",
                link: post.link || "",
                title: {
                    rendered: ((_a = post.title) === null || _a === void 0 ? void 0 : _a.rendered) || post.title || "",
                    raw: (_b = post.title) === null || _b === void 0 ? void 0 : _b.raw,
                    protected: ((_c = post.title) === null || _c === void 0 ? void 0 : _c.protected) || false
                },
                content: {
                    rendered: ((_d = post.content) === null || _d === void 0 ? void 0 : _d.rendered) || post.content || "",
                    raw: (_e = post.content) === null || _e === void 0 ? void 0 : _e.raw,
                    protected: ((_f = post.content) === null || _f === void 0 ? void 0 : _f.protected) || false,
                    block_version: (_g = post.content) === null || _g === void 0 ? void 0 : _g.block_version
                },
                excerpt: {
                    rendered: ((_h = post.excerpt) === null || _h === void 0 ? void 0 : _h.rendered) || post.excerpt || "",
                    raw: (_j = post.excerpt) === null || _j === void 0 ? void 0 : _j.raw,
                    protected: ((_k = post.excerpt) === null || _k === void 0 ? void 0 : _k.protected) || false
                },
                guid: {
                    rendered: ((_l = post.guid) === null || _l === void 0 ? void 0 : _l.rendered) || post.guid || "",
                    raw: (_m = post.guid) === null || _m === void 0 ? void 0 : _m.raw
                },
                author: post.author || 0,
                featured_media: post.featured_media || 0,
                comment_status: post.comment_status || "open",
                ping_status: post.ping_status || "open",
                ...post
            };
            return normalized;
        }
        /**
         * Normalize a term object to match the generic schema
         * @param term - Term object to normalize
         * @returns Normalized term object
         */
        static normalizeTerm(term) {
            const normalized = {
                id: term.id || 0,
                count: term.count || 0,
                description: term.description || "",
                link: term.link || "",
                name: term.name || "",
                slug: term.slug || "",
                taxonomy: term.taxonomy || "",
                parent: term.parent || 0,
                ...term
            };
            return normalized;
        }
        /**
         * Normalize a taxonomy object to match the generic schema
         * @param taxonomy - Taxonomy object to normalize
         * @returns Normalized taxonomy object
         */
        static normalizeTaxonomy(taxonomy) {
            var _a, _b;
            const normalized = {
                name: taxonomy.name || "",
                slug: taxonomy.slug || "",
                description: taxonomy.description || "",
                types: taxonomy.types || [],
                hierarchical: taxonomy.hierarchical || false,
                rest_base: taxonomy.rest_base || "",
                rest_controller_class: taxonomy.rest_controller_class || "",
                labels: {
                    name: ((_a = taxonomy.labels) === null || _a === void 0 ? void 0 : _a.name) || "",
                    singular_name: ((_b = taxonomy.labels) === null || _b === void 0 ? void 0 : _b.singular_name) || "",
                    ...taxonomy.labels
                },
                ...taxonomy
            };
            return normalized;
        }
    }

    // Core exports
    // Environment detection utilities (simple checks)
    const isBrowser = () => typeof window !== 'undefined' && typeof document !== 'undefined';
    // Convenience function for quick setup
    function createWordPressClient(baseUrl, cache) {
        return new WordPress({
            baseUrl,
            cache
        });
    }
    // Convenience function with authentication
    function createWordPressClientWithAuth(baseUrl, username, password, cache) {
        return new WordPress({
            baseUrl,
            auth: { username, password },
            cache
        });
    }
    // Convenience function with token
    function createWordPressClientWithToken(baseUrl, token, cache) {
        return new WordPress({
            baseUrl,
            auth: { token },
            cache
        });
    }
    // Enhanced convenience function with environment-aware defaults
    function createWordPressClientWithDefaults(baseUrl, options = {}) {
        return new WordPress({
            baseUrl,
            auth: options.auth,
            cache: options.cache
        });
    }

    exports.POST_TYPE_MAPPINGS = POST_TYPE_MAPPINGS;
    exports.TAXONOMY_MAPPINGS = TAXONOMY_MAPPINGS;
    exports.WPCache = WPCache;
    exports.WPClient = WPClient;
    exports.WPGenericPostJSONSchema = WPGenericPostJSONSchema;
    exports.WPGenericTaxonomyJSONSchema = WPGenericTaxonomyJSONSchema;
    exports.WPGenericTermJSONSchema = WPGenericTermJSONSchema;
    exports.WPMenusAPI = WPMenusAPI;
    exports.WPMetaAPI = WPMetaAPI;
    exports.WPPostsAPI = WPPostsAPI;
    exports.WPSchemaValidator = WPSchemaValidator;
    exports.WPSimplePostJSONSchema = WPSimplePostJSONSchema;
    exports.WPSimpleTaxonomyJSONSchema = WPSimpleTaxonomyJSONSchema;
    exports.WPSimpleTermJSONSchema = WPSimpleTermJSONSchema;
    exports.WPTaxonomiesAPI = WPTaxonomiesAPI;
    exports.WordPress = WordPress;
    exports.createWordPressClient = createWordPressClient;
    exports.createWordPressClientWithAuth = createWordPressClientWithAuth;
    exports.createWordPressClientWithDefaults = createWordPressClientWithDefaults;
    exports.createWordPressClientWithToken = createWordPressClientWithToken;
    exports.getAllPostTypeMappings = getAllPostTypeMappings;
    exports.getAllTaxonomyMappings = getAllTaxonomyMappings;
    exports.hasPostTypeMapping = hasPostTypeMapping;
    exports.hasTaxonomyMapping = hasTaxonomyMapping;
    exports.isBrowser = isBrowser;
    exports.mapPostTypeToEndpoint = mapPostTypeToEndpoint;
    exports.mapTaxonomyToEndpoint = mapTaxonomyToEndpoint;
    exports.registerPostTypeMapping = registerPostTypeMapping;
    exports.registerTaxonomyMapping = registerTaxonomyMapping;

}));
