<?php
/**
 * Post Operations REST API Class
 *
 * Provides REST API endpoints for creating, retrieving, updating, and deleting WordPress posts,
 * categories, and tags.
 *
 * @package    WVC_Theme
 * @subpackage REST_API
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 */

// Prevent direct access
if ( ! defined("ABSPATH") ) {
    exit;
}

require_once get_template_directory() . '/includes/rest/wvc-rest-api.php';
require_once get_template_directory() . '/includes/content-managers/post-manager.php';

/**
 * Class WVC_Post_REST_API
 *
 * Handles post, category, and tag operations via REST API
 */
class WVC_Post_REST_API extends WVC_REST_API {

    /**
     * Post Manager instance
     *
     * @var WVC_Post_Manager
     */
    private $post_manager;

    /**
     * Required capability for posts management
     */
    const EDIT_POSTS_CAP = 'edit_posts';

    /**
     * Required capability for managing categories and tags
     */
    const MANAGE_CATEGORIES_CAP = 'manage_categories';

    /**
     * Constructor
     */
    public function __construct() {
        parent::__construct();
        $this->post_manager = WVC_Post_Manager::get_instance();
    }

    /**
     * Required capability for media management
     */
    const UPLOAD_FILES_CAP = 'upload_files';

    /**
     * Register REST API routes
     */
    public function register_routes() {
        // Register post endpoints
        $this->register_post_routes();
        
        // Register category endpoints
        $this->register_category_routes();
        
        // Register tag endpoints
        $this->register_tag_routes();
        
        // Register media endpoints
        $this->register_media_routes();
    }

    /**
     * Register post-related routes
     */
    private function register_post_routes() {
        // Create post endpoint
        register_rest_route(
            $this->namespace,
            '/posts',
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array( $this, 'create_post' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::EDIT_POSTS_CAP );
                },
                'args'                => array(
                    'title'           => array(
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __( 'The title of the post', 'wvc-theme' ),
                    ),
                    'content'         => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __( 'The content of the post', 'wvc-theme' ),
                    ),
                    'excerpt'         => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __( 'The excerpt of the post', 'wvc-theme' ),
                    ),
                    'status'          => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'default'           => 'draft',
                        'enum'              => array( 'publish', 'draft', 'pending', 'private' ),
                        'description'       => __( 'The status of the post', 'wvc-theme' ),
                    ),
                    'author'          => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'description' => __( 'The author ID of the post', 'wvc-theme' ),
                    ),
                    'categories'      => array(
                        'required'    => false,
                        'type'        => 'array',
                        'items'       => array(
                            'type' => 'integer',
                        ),
                        'description' => __( 'Array of category IDs', 'wvc-theme' ),
                    ),
                    'tags'            => array(
                        'required'    => false,
                        'type'        => 'array',
                        'items'       => array(
                            'type' => 'integer',
                        ),
                        'description' => __( 'Array of tag IDs', 'wvc-theme' ),
                    ),
                    'meta'            => array(
                        'required'    => false,
                        'type'        => 'object',
                        'description' => __( 'Meta data for the post', 'wvc-theme' ),
                    ),
                    'featured_image'  => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'description' => __( 'Featured image ID for the post', 'wvc-theme' ),
                    ),
                    'comment_status'  => array(
                        'required'    => false,
                        'type'        => 'string',
                        'enum'        => array( 'open', 'closed' ),
                        'description' => __( 'Comment status for the post', 'wvc-theme' ),
                    ),
                    'ping_status'     => array(
                        'required'    => false,
                        'type'        => 'string',
                        'enum'        => array( 'open', 'closed' ),
                        'description' => __( 'Ping status for the post', 'wvc-theme' ),
                    ),
                    'post_type'       => array(
                        'required'    => false,
                        'type'        => 'string',
                        'default'     => 'post',
                        'description' => __( 'The post type (e.g., "post", "testimonial", "product")', 'wvc-theme' ),
                    ),
                ),
            )
        );

        // Get post endpoint
        register_rest_route(
            $this->namespace,
            '/posts/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array( $this, 'get_post' ),
                'permission_callback' => '__return_true', // Public access for reading
                'args'                => array(
                    'id' => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the post to retrieve', 'wvc-theme' ),
                    ),
                ),
            )
        );

        // Update post endpoint
        register_rest_route(
            $this->namespace,
            '/posts/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => array( $this, 'update_post' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::EDIT_POSTS_CAP );
                },
                'args'                => array(
                    'id'              => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the post to update', 'wvc-theme' ),
                    ),
                    'title'           => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __( 'The title of the post', 'wvc-theme' ),
                    ),
                    'content'         => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __( 'The content of the post', 'wvc-theme' ),
                    ),
                    'excerpt'         => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __( 'The excerpt of the post', 'wvc-theme' ),
                    ),
                    'status'          => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'enum'              => array( 'publish', 'draft', 'pending', 'private' ),
                        'description'       => __( 'The status of the post', 'wvc-theme' ),
                    ),
                    'author'          => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'description' => __( 'The author ID of the post', 'wvc-theme' ),
                    ),
                    'categories'      => array(
                        'required'    => false,
                        'type'        => 'array',
                        'items'       => array(
                            'type' => 'integer',
                        ),
                        'description' => __( 'Array of category IDs', 'wvc-theme' ),
                    ),
                    'tags'            => array(
                        'required'    => false,
                        'type'        => 'array',
                        'items'       => array(
                            'type' => 'integer',
                        ),
                        'description' => __( 'Array of tag IDs', 'wvc-theme' ),
                    ),
                    'meta'            => array(
                        'required'    => false,
                        'type'        => 'object',
                        'description' => __( 'Meta data for the post', 'wvc-theme' ),
                    ),
                    'featured_image'  => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'description' => __( 'Featured image ID for the post', 'wvc-theme' ),
                    ),
                    'comment_status'  => array(
                        'required'    => false,
                        'type'        => 'string',
                        'enum'        => array( 'open', 'closed' ),
                        'description' => __( 'Comment status for the post', 'wvc-theme' ),
                    ),
                    'ping_status'     => array(
                        'required'    => false,
                        'type'        => 'string',
                        'enum'        => array( 'open', 'closed' ),
                        'description' => __( 'Ping status for the post', 'wvc-theme' ),
                    ),
                ),
            )
        );

        // Delete post endpoint
        register_rest_route(
            $this->namespace,
            '/posts/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::DELETABLE,
                'callback'            => array( $this, 'delete_post' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::EDIT_POSTS_CAP );
                },
                'args'                => array(
                    'id'    => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the post to delete', 'wvc-theme' ),
                    ),
                    'force' => array(
                        'required'    => false,
                        'type'        => 'boolean',
                        'default'     => false,
                        'description' => __( 'Whether to bypass trash and force deletion', 'wvc-theme' ),
                    ),
                ),
            )
        );
    }

    /**
     * Register category-related routes
     */
    private function register_category_routes() {
        // Create category endpoint
        register_rest_route(
            $this->namespace,
            '/categories',
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array( $this, 'create_category' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::MANAGE_CATEGORIES_CAP );
                },
                'args'                => array(
                    'name'        => array(
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __( 'The name of the category', 'wvc-theme' ),
                    ),
                    'slug'        => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_title',
                        'description'       => __( 'The slug of the category', 'wvc-theme' ),
                    ),
                    'description' => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __( 'The description of the category', 'wvc-theme' ),
                    ),
                    'parent'      => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'default'     => 0,
                        'description' => __( 'The parent ID of the category', 'wvc-theme' ),
                    ),
                    'meta'        => array(
                        'required'    => false,
                        'type'        => 'object',
                        'description' => __( 'Meta data for the category', 'wvc-theme' ),
                    ),
                ),
            )
        );

        // Get category endpoint
        register_rest_route(
            $this->namespace,
            '/categories/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array( $this, 'get_category' ),
                'permission_callback' => '__return_true', // Public access for reading
                'args'                => array(
                    'id' => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the category to retrieve', 'wvc-theme' ),
                    ),
                ),
            )
        );

        // Update category endpoint
        register_rest_route(
            $this->namespace,
            '/categories/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => array( $this, 'update_category' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::MANAGE_CATEGORIES_CAP );
                },
                'args'                => array(
                    'id'          => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the category to update', 'wvc-theme' ),
                    ),
                    'name'        => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __( 'The name of the category', 'wvc-theme' ),
                    ),
                    'slug'        => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_title',
                        'description'       => __( 'The slug of the category', 'wvc-theme' ),
                    ),
                    'description' => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __( 'The description of the category', 'wvc-theme' ),
                    ),
                    'parent'      => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'description' => __( 'The parent ID of the category', 'wvc-theme' ),
                    ),
                    'meta'        => array(
                        'required'    => false,
                        'type'        => 'object',
                        'description' => __( 'Meta data for the category', 'wvc-theme' ),
                    ),
                ),
            )
        );

        // Delete category endpoint
        register_rest_route(
            $this->namespace,
            '/categories/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::DELETABLE,
                'callback'            => array( $this, 'delete_category' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::MANAGE_CATEGORIES_CAP );
                },
                'args'                => array(
                    'id' => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the category to delete', 'wvc-theme' ),
                    ),
                ),
            )
        );
    }

    /**
     * Register tag-related routes
     */
    private function register_tag_routes() {
        // Create tag endpoint
        register_rest_route(
            $this->namespace,
            '/tags',
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array( $this, 'create_tag' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::MANAGE_CATEGORIES_CAP );
                },
                'args'                => array(
                    'name'        => array(
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __( 'The name of the tag', 'wvc-theme' ),
                    ),
                    'slug'        => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_title',
                        'description'       => __( 'The slug of the tag', 'wvc-theme' ),
                    ),
                    'description' => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __( 'The description of the tag', 'wvc-theme' ),
                    ),
                    'meta'        => array(
                        'required'    => false,
                        'type'        => 'object',
                        'description' => __( 'Meta data for the tag', 'wvc-theme' ),
                    ),
                ),
            )
        );

        // Get tag endpoint
        register_rest_route(
            $this->namespace,
            '/tags/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array( $this, 'get_tag' ),
                'permission_callback' => '__return_true', // Public access for reading
                'args'                => array(
                    'id' => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the tag to retrieve', 'wvc-theme' ),
                    ),
                ),
            )
        );

        // Update tag endpoint
        register_rest_route(
            $this->namespace,
            '/tags/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => array( $this, 'update_tag' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::MANAGE_CATEGORIES_CAP );
                },
                'args'                => array(
                    'id'          => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the tag to update', 'wvc-theme' ),
                    ),
                    'name'        => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __( 'The name of the tag', 'wvc-theme' ),
                    ),
                    'slug'        => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_title',
                        'description'       => __( 'The slug of the tag', 'wvc-theme' ),
                    ),
                    'description' => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __( 'The description of the tag', 'wvc-theme' ),
                    ),
                    'meta'        => array(
                        'required'    => false,
                        'type'        => 'object',
                        'description' => __( 'Meta data for the tag', 'wvc-theme' ),
                    ),
                ),
            )
        );

        // Delete tag endpoint
        register_rest_route(
            $this->namespace,
            '/tags/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::DELETABLE,
                'callback'            => array( $this, 'delete_tag' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::MANAGE_CATEGORIES_CAP );
                },
                'args'                => array(
                    'id' => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the tag to delete', 'wvc-theme' ),
                    ),
                ),
            )
        );
    }

    /**
     * Create a new post
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function create_post( $request ) {
        $args = array(
            'title'          => $request->get_param( 'title' ),
            'content'        => $request->get_param( 'content' ),
            'excerpt'        => $request->get_param( 'excerpt' ),
            'status'         => $request->get_param( 'status' ),
            'author'         => $request->get_param( 'author' ),
            'categories'     => $request->get_param( 'categories' ),
            'tags'           => $request->get_param( 'tags' ),
            'meta'           => $request->get_param( 'meta' ),
            'featured_image' => $request->get_param( 'featured_image' ),
            'comment_status' => $request->get_param( 'comment_status' ),
            'ping_status'    => $request->get_param( 'ping_status' ),
            'post_type'      => $request->get_param( 'post_type' ),
        );

        $result = $this->post_manager->create_post( $args );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Get a post
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function get_post( $request ) {
        $post_id = $request->get_param( 'id' );
        $result  = $this->post_manager->get_post( $post_id );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Update a post
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function update_post( $request ) {
        $post_id = $request->get_param( 'id' );
        $args    = array(
            'title'          => $request->get_param( 'title' ),
            'content'        => $request->get_param( 'content' ),
            'excerpt'        => $request->get_param( 'excerpt' ),
            'status'         => $request->get_param( 'status' ),
            'author'         => $request->get_param( 'author' ),
            'categories'     => $request->get_param( 'categories' ),
            'tags'           => $request->get_param( 'tags' ),
            'meta'           => $request->get_param( 'meta' ),
            'featured_image' => $request->get_param( 'featured_image' ),
            'comment_status' => $request->get_param( 'comment_status' ),
            'ping_status'    => $request->get_param( 'ping_status' ),
        );

        $result = $this->post_manager->update_post( $post_id, $args );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Delete a post
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function delete_post( $request ) {
        $post_id = $request->get_param( 'id' );
        $force   = $request->get_param( 'force' );

        $result = $this->post_manager->delete_post( $post_id, $force );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Create a new category
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function create_category( $request ) {
        $args = array(
            'name'        => $request->get_param( 'name' ),
            'slug'        => $request->get_param( 'slug' ),
            'description' => $request->get_param( 'description' ),
            'parent'      => $request->get_param( 'parent' ),
            'meta'        => $request->get_param( 'meta' ),
        );

        $result = $this->post_manager->create_category( $args );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Get a category
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function get_category( $request ) {
        $category_id = $request->get_param( 'id' );
        $result      = $this->post_manager->get_category( $category_id );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Update a category
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function update_category( $request ) {
        $category_id = $request->get_param( 'id' );
        $args        = array(
            'name'        => $request->get_param( 'name' ),
            'slug'        => $request->get_param( 'slug' ),
            'description' => $request->get_param( 'description' ),
            'parent'      => $request->get_param( 'parent' ),
            'meta'        => $request->get_param( 'meta' ),
        );

        $result = $this->post_manager->update_category( $category_id, $args );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Delete a category
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function delete_category( $request ) {
        $category_id = $request->get_param( 'id' );
        $result      = $this->post_manager->delete_category( $category_id );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Create a new tag
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function create_tag( $request ) {
        $args = array(
            'name'        => $request->get_param( 'name' ),
            'slug'        => $request->get_param( 'slug' ),
            'description' => $request->get_param( 'description' ),
            'meta'        => $request->get_param( 'meta' ),
        );

        $result = $this->post_manager->create_tag( $args );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Get a tag
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function get_tag( $request ) {
        $tag_id = $request->get_param( 'id' );
        $result = $this->post_manager->get_tag( $tag_id );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Update a tag
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function update_tag( $request ) {
        $tag_id = $request->get_param( 'id' );
        $args   = array(
            'name'        => $request->get_param( 'name' ),
            'slug'        => $request->get_param( 'slug' ),
            'description' => $request->get_param( 'description' ),
            'meta'        => $request->get_param( 'meta' ),
        );

        $result = $this->post_manager->update_tag( $tag_id, $args );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }

    /**
     * Delete a tag
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function delete_tag( $request ) {
        $tag_id = $request->get_param( 'id' );
        $result = $this->post_manager->delete_tag( $tag_id );

        if ( is_wp_error( $result ) ) {
            return $result;
        }

        return rest_ensure_response( $result );
    }
    
    /**
     * Register media-related routes
     */
    private function register_media_routes() {
        // Upload media endpoint
        register_rest_route(
            $this->namespace,
            '/media',
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array( $this, 'upload_media' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::UPLOAD_FILES_CAP );
                },
                'args'                => array(
                    'post_id'     => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'description' => __( 'The post ID to attach the media to', 'wvc-theme' ),
                        'default'     => 0,
                    ),
                    'title'       => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __( 'The title of the media', 'wvc-theme' ),
                    ),
                    'caption'     => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __( 'The caption of the media', 'wvc-theme' ),
                    ),
                    'description' => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __( 'The description of the media', 'wvc-theme' ),
                    ),
                    'alt_text'    => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __( 'The alternative text for the media', 'wvc-theme' ),
                    ),
                ),
            )
        );
        
        // Get media endpoint
        register_rest_route(
            $this->namespace,
            '/media/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array( $this, 'get_media' ),
                'permission_callback' => '__return_true', // Public access for reading
                'args'                => array(
                    'id' => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the media to retrieve', 'wvc-theme' ),
                    ),
                ),
            )
        );
        
        // Update media endpoint
        register_rest_route(
            $this->namespace,
            '/media/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => array( $this, 'update_media' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::UPLOAD_FILES_CAP );
                },
                'args'                => array(
                    'id'          => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the media to update', 'wvc-theme' ),
                    ),
                    'title'       => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __( 'The title of the media', 'wvc-theme' ),
                    ),
                    'caption'     => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __( 'The caption of the media', 'wvc-theme' ),
                    ),
                    'description' => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __( 'The description of the media', 'wvc-theme' ),
                    ),
                    'alt_text'    => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __( 'The alternative text for the media', 'wvc-theme' ),
                    ),
                ),
            )
        );
        
        // Delete media endpoint
        register_rest_route(
            $this->namespace,
            '/media/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::DELETABLE,
                'callback'            => array( $this, 'delete_media' ),
                'permission_callback' => function ( WP_REST_Request $request ) {
                    return $this->permissions_check( $request, self::UPLOAD_FILES_CAP );
                },
                'args'                => array(
                    'id'    => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __( 'The ID of the media to delete', 'wvc-theme' ),
                    ),
                    'force' => array(
                        'required'    => false,
                        'type'        => 'boolean',
                        'default'     => false,
                        'description' => __( 'Whether to bypass trash and force deletion', 'wvc-theme' ),
                    ),
                ),
            )
        );
    }
    
    /**
     * Upload media
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function upload_media($request) {
        // Handle binary upload from Python client (raw data)
        $content_type = $request->get_header('content-type');
        $content_disposition = $request->get_header('content-disposition');
        
        // Check if we have proper headers for binary upload
        if (empty($content_disposition)) {
            return new WP_Error(
                'missing_headers',
                __('Missing Content-Disposition header', 'wvc-theme'),
                array('status' => 400)
            );
        }
        
        // Parse filename from Content-Disposition header
        $filename = '';
        if (preg_match('/filename="([^"]+)"/', $content_disposition, $matches)) {
            $filename = $matches[1];
        } else {
            return new WP_Error(
                'invalid_filename',
                __('Invalid or missing filename in Content-Disposition header', 'wvc-theme'),
                array('status' => 400)
            );
        }
        
        // Get the raw data from the request
        $image_data = $request->get_body();
        
        if (empty($image_data)) {
            return new WP_Error(
                'empty_data',
                __('No data was provided in the request body', 'wvc-theme'),
                array('status' => 400)
            );
        }
        
        // Get the post ID if provided
        $post_id = $request->get_param('post_id');
        
        // Set up arguments for media upload
        $args = array(
            'title'       => $request->get_param('title'),
            'caption'     => $request->get_param('caption'),
            'description' => $request->get_param('description'),
            'alt_text'    => $request->get_param('alt_text'),
        );
        
        // Use the refactored upload_media method
        $result = $this->post_manager->upload_media(
            $image_data,
            $filename,
            $content_type,
            $post_id,
            $args
        );
        
        if (is_wp_error($result)) {
            return $result;
        }
        
        return rest_ensure_response($result);
    }
    
    /**
     * Get media
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function get_media( $request ) {
        $media_id = $request->get_param( 'id' );
        $result   = $this->post_manager->get_media( $media_id );
        
        if ( is_wp_error( $result ) ) {
            return $result;
        }
        
        return rest_ensure_response( $result );
    }
    
    /**
     * Update media
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function update_media( $request ) {
        $media_id = $request->get_param( 'id' );
        $args     = array(
            'title'       => $request->get_param( 'title' ),
            'caption'     => $request->get_param( 'caption' ),
            'description' => $request->get_param( 'description' ),
            'alt_text'    => $request->get_param( 'alt_text' ),
        );
        
        $result = $this->post_manager->update_media( $media_id, $args );
        
        if ( is_wp_error( $result ) ) {
            return $result;
        }
        
        return rest_ensure_response( $result );
    }
    
    /**
     * Delete media
     *
     * @param WP_REST_Request $request The request object
     *
     * @return WP_REST_Response|WP_Error
     */
    public function delete_media( $request ) {
        $media_id = $request->get_param( 'id' );
        $force    = $request->get_param( 'force' );
        
        $result = $this->post_manager->delete_media( $media_id, $force );
        
        if ( is_wp_error( $result ) ) {
            return $result;
        }
        
        return rest_ensure_response( $result );
    }
}

// Initialize the Post Operations REST API
new WVC_Post_REST_API();