<?php
/**
 * Menu Operations REST API Class
 *
 * Provides REST API endpoints for creating and deleting WordPress menus.
 *
 * @package    WVC_Theme
 * @subpackage REST_API
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 */

// Prevent direct access
if ( ! defined("ABSPATH")) {
    exit;
}

require_once get_template_directory() . '/includes/rest/wvc-rest-api.php';
require_once get_template_directory() . '/includes/content-managers/menu-manager.php';

/**
 * Class WVC_Menu_REST_API
 *
 * Handles menu creation and deletion via REST API
 */
class WVC_Menu_REST_API extends WVC_REST_API
{

    /**
     * Menu Manager instance
     *
     * @var WVC_Menu_Manager
     */
    private $menu_manager;

    const EDIT_THEME_OPTIONS_CAP = "edit_theme_options";

    /**
     * Constructor
     */
    public function __construct()
    {
        parent::__construct();
        $this->menu_manager = WVC_Menu_Manager::get_instance();
    }

    /**
     * Register REST API routes
     */
    public function register_routes()
    {
        // Get all menus endpoint
        register_rest_route(
            $this->namespace,
            '/menus',
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array($this, 'get_menus'),
                'permission_callback' => '__return_true', // Public access for reading
            )
        );

        // Create menu endpoint
        register_rest_route(
            $this->namespace,
            '/menus',
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array($this, 'create_menu'),
                'permission_callback' => function (WP_REST_Request $request) {
                    return $this->permissions_check($request, self::EDIT_THEME_OPTIONS_CAP);
                },
                'args'                => array(
                    'name'     => array(
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __('The name of the menu to create', 'wvc-theme'),
                    ),
                    'location' => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __('Optional menu location to assign the menu to', 'wvc-theme'),
                    ),
                    'items'    => array(
                        'required'    => false,
                        'type'        => 'array',
                        'description' => __('Optional array of menu items to add', 'wvc-theme'),
                        'items'       => array(
                            'type'       => 'object',
                            'properties' => array(
                                'title'       => array(
                                    'type'        => 'string',
                                    'description' => __('Title of the menu item', 'wvc-theme'),
                                ),
                                'url'         => array(
                                    'type'        => 'string',
                                    'description' => __('URL for the menu item', 'wvc-theme'),
                                ),
                                'object_id'   => array(
                                    'type'        => 'integer',
                                    'description' => __('ID of the object (post, page, etc.) to link to', 'wvc-theme'),
                                ),
                                'object_type' => array(
                                    'type'        => 'string',
                                    'description' => __('Type of object (post_type, taxonomy, custom)', 'wvc-theme'),
                                ),
                                'parent'      => array(
                                    'type'        => 'integer',
                                    'description' => __('ID of the parent menu item', 'wvc-theme'),
                                ),
                                'order'       => array(
                                    'type'        => 'integer',
                                    'description' => __('Order of the menu item', 'wvc-theme'),
                                ),
                            ),
                        ),
                    ),
                ),
            )
        );

        // Get single menu with items endpoint
        register_rest_route(
            $this->namespace,
            '/menus/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array($this, 'get_menu'),
                'permission_callback' => '__return_true',
                'args'                => array(
                    'id' => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __('The ID of the menu to retrieve', 'wvc-theme'),
                    ),
                ),
            )
        );

        // Delete menu endpoint
        register_rest_route(
            $this->namespace,
            '/menus/(?P<id>\d+)',
            array(
                'methods'             => WP_REST_Server::DELETABLE,
                'callback'            => array($this, 'delete_menu'),
                'permission_callback' => function (WP_REST_Request $request)  {
                    return $this->permissions_check($request, self::EDIT_THEME_OPTIONS_CAP);
                },
                'args'                => array(
                    'id' => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __('The ID of the menu to delete', 'wvc-theme'),
                    ),
                ),
            )
        );

        // Add menu item endpoint
        register_rest_route(
            $this->namespace,
            '/menus/(?P<menu_id>\d+)/items',
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array($this, 'add_menu_item_endpoint'),
                'permission_callback' =>  function (WP_REST_Request $request) {
                    return $this->permissions_check($request, self::EDIT_THEME_OPTIONS_CAP);
                },
                'args'                => array(
                    'menu_id'     => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __('The ID of the menu', 'wvc-theme'),
                    ),
                    'title'       => array(
                        'required'          => true,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __('Title of the menu item', 'wvc-theme'),
                    ),
                    'url'         => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __('URL for the menu item (for custom links)', 'wvc-theme'),
                    ),
                    'object_id'   => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'description' => __('ID of the object (post, page, etc.) to link to', 'wvc-theme'),
                    ),
                    'object_type' => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __('Type of object (post_type, taxonomy, custom)', 'wvc-theme'),
                    ),
                    'parent'      => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'description' => __('ID of the parent menu item', 'wvc-theme'),
                    ),
                    'order'       => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'description' => __('Order of the menu item', 'wvc-theme'),
                    ),
                ),
            )
        );

        // Edit menu item endpoint
        register_rest_route(
            $this->namespace,
            '/menus/(?P<menu_id>\d+)/items/(?P<item_id>\d+)',
            array(
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => array($this, 'edit_menu_item'),
                'permission_callback' => function (WP_REST_Request $request) {
                    return $this->permissions_check($request, self::EDIT_THEME_OPTIONS_CAP);
                },
                'args'                => array(
                    'menu_id'     => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __('The ID of the menu', 'wvc-theme'),
                    ),
                    'item_id'     => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __('The ID of the menu item to edit', 'wvc-theme'),
                    ),
                    'title'       => array(
                        'required'          => false,
                        'type'              => 'string',
                        'sanitize_callback' => 'sanitize_text_field',
                        'description'       => __('The title of the menu item', 'wvc-theme'),
                    ),
                    'url'         => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __('URL for the menu item (for custom links)', 'wvc-theme'),
                    ),
                    'object_id'   => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'description' => __('ID of the object (post, page, etc.) to link to', 'wvc-theme'),
                    ),
                    'object_type' => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __('Type of object (post_type, taxonomy, custom)', 'wvc-theme'),
                    ),
                    'parent'      => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'description' => __('ID of the parent menu item', 'wvc-theme'),
                    ),
                    'order'       => array(
                        'required'    => false,
                        'type'        => 'integer',
                        'description' => __('Order of the menu item', 'wvc-theme'),
                    ),
                    'target'      => array(
                        'required'    => false,
                        'type'        => 'string',
                        'enum'        => array('', '_blank', '_self', '_parent', '_top'),
                        'description' => __('Target for the menu item link', 'wvc-theme'),
                    ),
                    'classes'     => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __('CSS classes for the menu item', 'wvc-theme'),
                    ),
                    'description' => array(
                        'required'    => false,
                        'type'        => 'string',
                        'description' => __('Description for the menu item', 'wvc-theme'),
                    ),
                ),
            )
        );

        // Delete menu item endpoint
        register_rest_route(
            $this->namespace,
            '/menus/(?P<menu_id>\d+)/items/(?P<item_id>\d+)',
            array(
                'methods'             => WP_REST_Server::DELETABLE,
                'callback'            => array($this, 'delete_menu_item'),
                'permission_callback' => function (WP_REST_Request $request) {
                    return $this->permissions_check($request, self::EDIT_THEME_OPTIONS_CAP);
                },
                'args'                => array(
                    'menu_id' => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __('The ID of the menu', 'wvc-theme'),
                    ),
                    'item_id' => array(
                        'required'    => true,
                        'type'        => 'integer',
                        'description' => __('The ID of the menu item to delete', 'wvc-theme'),
                    ),
                ),
            )
        );
    }

    /**
     * Create a new menu
     *
     * @param WP_REST_Request $request The request object.
     *
     * @return WP_REST_Response|WP_Error
     */
    public function create_menu($request)
    {
        $args = array(
            'name'     => $request->get_param('name'),
            'location' => $request->get_param('location'),
            'items'    => $request->get_param('items'),
        );

        $result = $this->menu_manager->create_menu($args);

        if (is_wp_error($result)) {
            return $result;
        }

        return rest_ensure_response($result);
    }

    /**
     * Add a menu item to a menu
     *
     * @param int $menu_id The menu ID
     * @param array $item The item data
     *
     * @return int|WP_Error  The menu item ID or WP_Error
     */
    private function add_menu_item($menu_id, $item)
    {
        // Default item args
        $args = array(
            'menu-item-title'  => isset($item['title']) ? $item['title'] : '',
            'menu-item-status' => 'publish',
        );

        // Set menu item type and object
        if ( ! empty($item['object_id']) && ! empty($item['object_type'])) {
            if ($item['object_type'] === 'post_type') {
                $post_type                   = get_post_type($item['object_id']);
                $args['menu-item-type']      = 'post_type';
                $args['menu-item-object']    = $post_type;
                $args['menu-item-object-id'] = $item['object_id'];
            } elseif ($item['object_type'] === 'taxonomy') {
                $term = get_term($item['object_id']);
                if ( ! is_wp_error($term)) {
                    $args['menu-item-type']      = 'taxonomy';
                    $args['menu-item-object']    = $term->taxonomy;
                    $args['menu-item-object-id'] = $item['object_id'];
                }
            }
        } elseif ( ! empty($item['url'])) {
            $args['menu-item-type'] = 'custom';
            $args['menu-item-url']  = esc_url_raw($item['url']);
        }

        // Set parent if provided
        if ( ! empty($item['parent'])) {
            $args['menu-item-parent-id'] = $item['parent'];
        }

        // Set order if provided
        if (isset($item['order'])) {
            $args['menu-item-position'] = $item['order'];
        }

        // Add the menu item
        return wp_update_nav_menu_item($menu_id, 0, $args);
    }

    /**
     * Delete a menu
     *
     * @param WP_REST_Request $request The request object.
     *
     * @return WP_REST_Response|WP_Error
     */
    public function delete_menu($request)
    {
        $menu_id = $request->get_param('id');
        $result  = $this->menu_manager->delete_menu($menu_id);

        if (is_wp_error($result)) {
            return $result;
        }

        return rest_ensure_response($result);
    }

    /**
     * Delete a menu item
     *
     * @param WP_REST_Request $request The request object.
     *
     * @return WP_REST_Response|WP_Error
     */
    /**
     * Get all menus
     *
     * @return WP_REST_Response
     */
    public function get_menus()
    {
        $result = $this->menu_manager->get_menus();

        return rest_ensure_response($result);
    }

    /**
     * Get a single menu with its items
     *
     * @param WP_REST_Request $request The request object.
     *
     * @return WP_REST_Response|WP_Error
     */
    public function get_menu($request)
    {
        $menu_id = $request->get_param('id');
        $result  = $this->menu_manager->get_menu($menu_id);

        if (is_wp_error($result)) {
            return $result;
        }

        return rest_ensure_response($result);
    }

    /**
     * Prepare menu response
     *
     * @param WP_Term $menu The menu object
     * @param array $menu_items Optional array of menu items
     *
     * @return array Formatted menu data
     */
    private function prepare_menu_response($menu, $menu_items = null)
    {
        $data = array(
            'id'          => $menu->term_id,
            'name'        => $menu->name,
            'slug'        => $menu->slug,
            'description' => $menu->description,
            'count'       => $menu->count,
        );

        // Get menu locations
        $locations      = get_nav_menu_locations();
        $menu_locations = array();

        foreach ($locations as $location => $menu_id) {
            if ($menu_id === $menu->term_id) {
                $menu_locations[] = $location;
            }
        }

        $data['locations'] = $menu_locations;

        // Add menu items if provided
        if ($menu_items) {
            $items = array();

            // Build hierarchical menu items array
            $menu_items_by_parent = array();
            foreach ($menu_items as $menu_item) {
                $parent = $menu_item->menu_item_parent ? $menu_item->menu_item_parent : 0;
                if ( ! isset($menu_items_by_parent[$parent])) {
                    $menu_items_by_parent[$parent] = array();
                }
                $menu_items_by_parent[$parent][] = $this->prepare_menu_item_response($menu_item);
            }

            // Get top level items
            if (isset($menu_items_by_parent[0])) {
                $items = $this->build_hierarchical_menu_items($menu_items_by_parent, 0);
            }

            $data['items'] = $items;
        }

        return $data;
    }

    /**
     * Build hierarchical menu items
     *
     * @param array $menu_items_by_parent Menu items grouped by parent
     * @param int $parent_id Parent ID
     *
     * @return array Hierarchical menu items
     */
    private function build_hierarchical_menu_items($menu_items_by_parent, $parent_id)
    {
        $items = array();

        if (isset($menu_items_by_parent[$parent_id])) {
            foreach ($menu_items_by_parent[$parent_id] as $item) {
                if (isset($menu_items_by_parent[$item['id']])) {
                    $item['children'] = $this->build_hierarchical_menu_items($menu_items_by_parent, $item['id']);
                }
                $items[] = $item;
            }
        }

        return $items;
    }

    /**
     * Prepare menu item response
     *
     * @param WP_Post $menu_item The menu item object
     *
     * @return array Formatted menu item data
     */
    private function prepare_menu_item_response($menu_item)
    {
        // Get menu item meta
        $menu_item_type = get_post_meta($menu_item->ID, '_menu_item_type', true);
        $object_id      = get_post_meta($menu_item->ID, '_menu_item_object_id', true);
        $object         = get_post_meta($menu_item->ID, '_menu_item_object', true);
        $url            = '';

        // Get URL based on item type
        if ($menu_item_type === 'custom') {
            $url = get_post_meta($menu_item->ID, '_menu_item_url', true);
        } elseif ($menu_item_type === 'post_type') {
            $url = get_permalink($object_id);
        } elseif ($menu_item_type === 'taxonomy') {
            $url = get_term_link((int)$object_id, $object);
            if (is_wp_error($url)) {
                $url = '';
            }
        }

        return array(
            'id'          => $menu_item->ID,
            'title'       => $menu_item->title,
            'url'         => $url,
            'type'        => $menu_item_type,
            'object_id'   => (int)$object_id,
            'object'      => $object,
            'parent'      => (int)$menu_item->menu_item_parent,
            'order'       => (int)$menu_item->menu_order,
            'target'      => get_post_meta($menu_item->ID, '_menu_item_target', true),
            'classes'     => implode(' ', (array)get_post_meta($menu_item->ID, '_menu_item_classes', true)),
            'xfn'         => get_post_meta($menu_item->ID, '_menu_item_xfn', true),
            'description' => $menu_item->description,
            'attr_title'  => $menu_item->attr_title,
        );
    }

    /**
     * Add a menu item via REST API
     *
     * @param WP_REST_Request $request The request object.
     *
     * @return WP_REST_Response|WP_Error
     */
    public function add_menu_item_endpoint($request)
    {
        $menu_id = $request->get_param('menu_id');
        $item    = array(
            'title'       => $request->get_param('title'),
            'url'         => $request->get_param('url'),
            'object_id'   => $request->get_param('object_id'),
            'object_type' => $request->get_param('object_type'),
            'parent'      => $request->get_param('parent'),
            'order'       => $request->get_param('order'),
            'target'      => $request->get_param('target'),
            'classes'     => $request->get_param('classes'),
            'description' => $request->get_param('description'),
        );

        $result = $this->menu_manager->add_menu_item($menu_id, $item);

        if (is_wp_error($result)) {
            return $result;
        }

        // Get the created menu item
        $menu_item = get_post($result);
        $item_data = $this->menu_manager->prepare_menu_item_response($menu_item);

        return rest_ensure_response(array(
            'success' => true,
            'menu_id' => $menu_id,
            'item'    => $item_data,
        ));
    }

    /**
     * Edit a menu item
     *
     * @param WP_REST_Request $request The request object.
     *
     * @return WP_REST_Response|WP_Error
     */
    public function edit_menu_item($request)
    {
        $menu_id = $request->get_param('menu_id');
        $item_id = $request->get_param('item_id');

        $args = array(
            'title'       => $request->get_param('title'),
            'url'         => $request->get_param('url'),
            'object_id'   => $request->get_param('object_id'),
            'object_type' => $request->get_param('object_type'),
            'parent'      => $request->get_param('parent'),
            'order'       => $request->get_param('order'),
            'target'      => $request->get_param('target'),
            'classes'     => $request->get_param('classes'),
            'description' => $request->get_param('description'),
        );

        $result = $this->menu_manager->edit_menu_item($menu_id, $item_id, $args);

        if (is_wp_error($result)) {
            return $result;
        }

        return rest_ensure_response($result);
    }

    /**
     * Delete a menu item
     *
     * @param WP_REST_Request $request The request object.
     *
     * @return WP_REST_Response|WP_Error
     */
    public function delete_menu_item($request)
    {
        $menu_id = $request->get_param('menu_id');
        $item_id = $request->get_param('item_id');

        $result = $this->menu_manager->delete_menu_item($menu_id, $item_id);

        if (is_wp_error($result)) {
            return $result;
        }

        return rest_ensure_response($result);
    }
}

// Initialize the Menu Operations REST API
new WVC_Menu_REST_API();