<?php

namespace WVC\FormSubmissions;

/**
 * WVC Form Submissions Handler
 *
 * Handles AJAX form submissions and data processing
 *
 * @package    WVC_Theme
 * @subpackage FormSubmissions
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.16
 */

// Prevent direct access
if ( ! defined("ABSPATH")) {
    exit;
}

/**
 * Handle form submission via AJAX
 */
function wvc_handle_ajax_form_submission() {
    // Verify nonce for security
    if (!check_ajax_referer('wvc_nonce', 'nonce', false)) {
        error_log("Invalid nonce in AJAX form submission");
        wp_send_json_error(array(
            'message' => 'Invalid security token'
        ), 403);
        return;
    }
    
    // Get parameters from $_POST
    $section_name = sanitize_text_field($_POST['sectionName'] ?? '');
    $form_id = sanitize_text_field($_POST['formId'] ?? '');
    $timestamp = floatval($_POST['timestamp'] ?? 0);
    $session_id = sanitize_text_field($_POST['sessionId'] ?? '');
    $page_url = esc_url_raw($_POST['pageUrl'] ?? '');
    $user_agent = sanitize_text_field($_POST['userAgent'] ?? '');
    $submission_attempt = intval($_POST['submissionAttempt'] ?? 1);
    $form_version = sanitize_text_field($_POST['formVersion'] ?? '1.0.0');
    $validation_errors = $_POST['validationErrors'] ?? '';
    
    // Validate required fields
    if (empty($section_name) || empty($form_id)) {
        wp_send_json_error(array(
            'message' => 'Missing required fields: sectionName and formId'
        ), 400);
        return;
    }
    
    // Extract and sanitize form data
    $form_data_raw = array();
    
    // Check if formData comes as an array
    if (isset($_POST['formData']) && is_array($_POST['formData'])) {
        foreach ($_POST['formData'] as $key => $value) {
            $form_data_raw[sanitize_key($key)] = sanitize_textarea_field($value);
        }
    } else {
        // Fallback: look for formData[key] pattern in $_POST
        foreach ($_POST as $key => $value) {
            if (strpos($key, 'formData[') === 0) {
                $field_name = str_replace(array('formData[', ']'), '', $key);
                $form_data_raw[sanitize_key($field_name)] = sanitize_textarea_field($value);
            }
        }
    }
    
    // Create post title with form ID and timestamp - sanitize for safety
    $post_title = sprintf('Form Submission: %s - %s', 
        sanitize_text_field($form_id), 
        sanitize_text_field(date('Y-m-d H:i:s', $timestamp / 1000))
    );
    
    // Sanitize all meta values before storing
    $sanitized_meta = array(
        '_wvc_form_id' => sanitize_text_field($form_id),
        '_wvc_section_name' => sanitize_text_field($section_name),
        '_wvc_timestamp' => floatval($timestamp), // Ensure numeric value
        '_wvc_session_id' => sanitize_text_field($session_id),
        '_wvc_page_url' => esc_url_raw($page_url),
        '_wvc_user_agent' => sanitize_textarea_field($user_agent), // Can be long, use textarea sanitization
        '_wvc_submission_attempt' => absint($submission_attempt), // Ensure positive integer
        '_wvc_form_version' => sanitize_text_field($form_version),
        '_wvc_validation_errors' => $validation_errors ? sanitize_textarea_field($validation_errors) : '',
        '_wvc_form_data' => wp_json_encode($form_data_raw, JSON_UNESCAPED_UNICODE) // Use wp_json_encode for better security
    );
    
    // Create the post
    $post_data = array(
        'post_title' => sanitize_text_field($post_title),
        'post_content' => '', // We'll store data in meta
        'post_status' => 'private',
        'post_type' => 'wvc_form_data',
        'post_author' => absint(get_current_user_id() ?: 1), // Use current user or fallback to admin
        'meta_input' => $sanitized_meta
    );
    
    // Insert the post
    $post_id = wp_insert_post($post_data, true);
    
    if (is_wp_error($post_id)) {
        wp_send_json_error(array(
            'message' => 'Failed to save form submission: ' . $post_id->get_error_message()
        ), 500);
        return;
    }
    
    // Store individual form fields as separate meta for easier querying
    foreach ($form_data_raw as $field_name => $field_value) {
        $sanitized_field_name = sanitize_key($field_name);
        $sanitized_field_value = sanitize_textarea_field($field_value);
        update_post_meta($post_id, '_wvc_field_' . $sanitized_field_name, $sanitized_field_value);
    }
    
    // Also store the complete form data structure in postmeta for backup/reference
    // Re-sanitize the complete form data array before storing
    $sanitized_form_data_complete = array();
    foreach ($form_data_raw as $key => $value) {
        $sanitized_form_data_complete[sanitize_key($key)] = sanitize_textarea_field($value);
    }
    update_post_meta($post_id, '_wvc_form_data_complete', $sanitized_form_data_complete);
    
    // Store additional metadata as individual post meta entries with sanitization
    $submission_metadata = array(
        'user_ip' => sanitize_text_field($_SERVER['REMOTE_ADDR'] ?? 'unknown'),
        'submission_source' => sanitize_text_field('website_form'),
        'browser_info' => sanitize_textarea_field($user_agent),
        'referrer' => esc_url_raw($_SERVER['HTTP_REFERER'] ?? ''),
        'form_fields_count' => absint(count($form_data_raw))
    );
    update_post_meta($post_id, '_wvc_submission_metadata', $submission_metadata);
    
    // Return success response using wp_send_json_success with sanitized values
    wp_send_json_success(array(
        'message' => sanitize_text_field('Form submission saved successfully'),
        'submission_id' => absint($post_id),
        'timestamp' => floatval($timestamp),
        'form_id' => sanitize_text_field($form_id)
    ));
}

/**
 * Register AJAX actions for form submission
 */
function wvc_register_ajax_actions() {
    // For logged-in users
    add_action('wp_ajax_wvc_form_submit', __NAMESPACE__ . '\wvc_handle_ajax_form_submission');
    // For non-logged-in users (public forms)
    add_action('wp_ajax_nopriv_wvc_form_submit', __NAMESPACE__ . '\wvc_handle_ajax_form_submission');
}