<?php

namespace WVC\Core;

/**
 * WVC Theme Core Functions
 *
 * This file contains core functionality for the WVC theme including
 * post meta registration and other essential features.
 *
 * @package    WVC_Theme
 * @subpackage Core
 * @author     10Web
 * @since      1.0.0
 * @version    1.0.0
 */

// Prevent direct access
if ( ! defined("ABSPATH")) {
    exit;
}

/**
 * WVC Core Class
 *
 * Handles core functionality for the WVC theme including
 * post meta registration and other essential features.
 */
class Core
{

    /**
     * Meta key for WVC editor
     */
    const WVC_EDITOR_META_KEY = '_wvc_editor';

    /**
     * Constructor
     */
    public function __construct()
    {
        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks()
    {
        // Hook the registration function to init
        add_action("init", array($this, "register_post_meta"));

        // AJAX handlers
        add_action("wp_ajax_wvc_ajax", array($this, "handle_ajax_request"));
        add_action("wp_ajax_nopriv_wvc", array($this, "handle_ajax_request"));
    }

    /**
     * Register post meta for page types
     *
     * Registers the wvc_editor meta field as a boolean with default value of false
     */
    public function register_post_meta()
    {
        register_post_meta(
            "page", // Object type - pages only
            self::WVC_EDITOR_META_KEY, // Meta key
            array(
                "type"          => "boolean", // Meta type
                "single"        => true, // Single meta value
                "default"       => false, // Default value (false as boolean)
                "show_in_rest"  => true, // Show in REST API
                "auth_callback" => function () {
                    return current_user_can("edit_posts"); // Only users who can edit posts can modify this meta
                }
            )
        );

        register_post_meta(
            "page", // Object type - pages only
            "_wvc_home", // Meta key
            array(
                "type"          => "boolean", // Meta type
                "single"        => true, // Single meta value
                "default"       => false, // Default value (false as boolean)
                "show_in_rest"  => true, // Show in REST API
                "auth_callback" => function () {
                    return current_user_can("edit_posts"); // Only users who can edit posts can modify this meta
                }
            )
        );
    }

    /**
     * Get WVC editor meta value for a post
     *
     * @param int $post_id Post ID
     *
     * @return bool WVC editor meta value
     */
    public static function get_wvc_editor_meta($post_id)
    {
        return (bool)get_post_meta($post_id, self::WVC_EDITOR_META_KEY, true);
    }

    /**
     * Set WVC editor meta value for a post
     *
     * @param int $post_id Post ID
     * @param bool $value WVC editor meta value
     *
     * @return bool|int Meta ID if the key didn't exist, true on successful update, false on failure
     */
    public static function set_wvc_editor_meta($post_id, $value)
    {
        return update_post_meta($post_id, self::WVC_EDITOR_META_KEY, (bool)$value);
    }

    /**
     * Delete WVC editor meta value for a post
     *
     * @param int $post_id Post ID
     *
     * @return bool True on successful delete, false on failure
     */
    public static function delete_wvc_editor_meta($post_id)
    {
        return delete_post_meta($post_id, self::WVC_EDITOR_META_KEY);
    }

    /**
     * Check if a post is built with WVC editor
     *
     * @param int $post_id Post ID
     *
     * @return bool True if built with WVC editor, false otherwise
     */
    public static function is_built_with_wvc($post_id)
    {
        return self::get_wvc_editor_meta($post_id);
    }


    public static function is_empty_vibecode_site()
    {
        // Check if wvc_empty option is set to 1
        $wvc_empty = get_option("wvc_empty", 0);

        if ($wvc_empty != 1) {
            return false;
        }

        // Check if there are no published posts with WVC editor meta key
        $published_posts = get_posts(array(
            'post_status'  => 'publish',
            'meta_key'     => self::WVC_EDITOR_META_KEY,
            'meta_value'   => true,
            'meta_compare' => '=',
            'numberposts'  => 1,
            'fields'       => 'ids'
        ));

        if ( ! empty($published_posts)) {
            return false;
        }

        return true;
    }


    /**
     * Mark the post as "built with WVC" or not
     *
     * @param int $post_id Post ID
     * @param bool $is_built_with_wvc Whether the post is built with WVC
     *
     * @return bool|int Meta ID if the key didn't exist, true on successful update, false on failure
     */
    public static function set_is_built_with_wvc($post_id, $is_built_with_wvc)
    {
        if ($is_built_with_wvc) {
            return self::set_wvc_editor_meta($post_id, true);
        } else {
            return self::delete_wvc_editor_meta($post_id);
        }
    }

    /**
     * Get 10Web-related configuration values
     *
     * @return array Array containing access token, user ID, domain ID, and workspace ID
     */
    public static function get_tenweb_config()
    {
        $config = array(
            'accessToken' => get_site_option('tenweb_access_token'),
            'wpUserId'    => get_current_user_id(),
            'domainId'    => get_site_option('tenweb_domain_id'),
            'workspaceId' => get_site_option('tenweb_workspace_id')
        );

        $tenweb_user_info = get_site_option('tenweb_user_info');
        if ($tenweb_user_info && isset($tenweb_user_info['agreement_info']['clients_id'])) {
            $config["clientId"] = $tenweb_user_info['agreement_info']['clients_id'];
        } else {
            $config["clientId"] = null;
        }

        return $config;
    }


    /**
     * Get WVC API URL with debug override support
     *
     * @return string|null The WVC API URL
     */
    public static function get_wvc_api_url()
    {
        // Check if debug mode is enabled and get the debug config
        $debug_mode = get_transient('wvc_debug_mode_enabled');
        if ($debug_mode) {
            $debug_configs = get_option('wvc_debug_configs', array());
            if ( ! empty($debug_configs['api_url'])) {
                return $debug_configs['api_url'] . "/api";
            }
        }

        // Fall back to default URL
        if (defined('TENWEB_WVC_URL')) {
            return TENWEB_WVC_URL . "/api";
        }

        return null;
    }

    public static function get_wvc_params_to_localize($post_id, $is_editor = false)
    {

        $wvc_url = self::get_wvc_api_url();


        $params =  array_merge(array(
            'ajaxurl' => admin_url('admin-ajax.php'),
            'resturl' => rest_url(),
            'nonce' => wp_create_nonce('wvc_nonce'),
            'postId' => $post_id,
            'postTitle' => $post_id ? get_the_title($post_id) : '',
            'postStatus' => $post_id ? get_post_status($post_id) : '',
            'isBuiltWithWvc' => $post_id ? \WVC\Core\Core::get_wvc_editor_meta($post_id) : false,
            'adminUrl'       => admin_url(),
            'wvcUrl'         => $wvc_url,
            "homeUrl"        => get_home_url(),
            'jsUtilsLink'    => get_template_directory_uri() . "/assets/js/index.umd.js",
            "siteTitle"      => get_bloginfo('name'),
            "dashboardUrl"   => TENWEB_DASHBOARD,
            "coreUrl"        => parse_url(TENWEB_SIGNUP_MAGIC_LINK_URL,
                    PHP_URL_SCHEME) . '://' . parse_url(TENWEB_SIGNUP_MAGIC_LINK_URL, PHP_URL_HOST)

        ), \WVC\Core\Core::get_tenweb_config());

        if ( ! $is_editor) {
            $params['nonce']        = null;
            $params['adminUrl']     = null;
            $params['dashboardUrl'] = null;
            $params['coreUrl']      = null;
            $params['accessToken']  = null;
            $params['wpUserId']     = null;
            $params['domainId']     = null;
            $params['workspaceId']  = null;
            $params['clientId']     = null;

        }

        return $params;

    }

    /**
     * AJAX handler for React components
     */
    public function handle_ajax_request()
    {
        check_ajax_referer("wvc_nonce", "nonce");

        $action = sanitize_text_field($_POST["action_type"]);

        switch ($action) {
            case "get_posts":
                $posts = get_posts(
                    array(
                        "numberposts" => 10,
                        "post_status" => "publish"
                    )
                );
                wp_send_json_success($posts);
                break;
            case "send_ga_event":
                $analytics_data = $_POST["data"];
                if (defined('TENWEB_SEND_GA_EVENT')) {
                    // Decode the escaped JSON string to get proper JSON
                    $decoded_data = stripslashes($analytics_data);
                    $body         = json_decode($decoded_data, true);
                    if ( ! is_array($body)) {
                        $body = array("data" => $decoded_data);
                    }
                    $tenweb_config = \WVC\Core\Core::get_tenweb_config();
                    unset($tenweb_config['accessToken']);

                    // Transform field names for GA event

                    $transformed_config = array();
                    if (isset($tenweb_config['domainId'])) {
                        $transformed_config['website_id'] = $tenweb_config['domainId'];
                    }
                    if (isset($tenweb_config['wpUserId'])) {
                        $transformed_config['wp_user_id'] = $tenweb_config['wpUserId'];
                    }
                    if (isset($tenweb_config['clientId'])) {
                        $transformed_config['tenweb_user_id'] = $tenweb_config['clientId'];
                    }
                    if (isset($tenweb_config['workspaceId'])) {
                        $transformed_config['workspace_id'] = $tenweb_config['workspaceId'];
                    }

                    if (is_array($body)) {

                        $body = array_merge($body, $transformed_config);
                    } else {
                        $body = $transformed_config;
                    }
                    $response = wp_remote_post(TENWEB_SEND_GA_EVENT, array(
                        'body'    => $body,
                        'headers' => array(
                            'Accept'       => 'application/x.10webcore.v1+json',
                            "Content-Type" => "application/x-www-form-urlencoded"
                        )
                    ));
                    if ( ! is_wp_error($response)) {
                        $response_code = wp_remote_retrieve_response_code($response);
                        if ($response_code >= 200 && $response_code < 300) {
                            wp_send_json_success();
                        } else {
                            $response_body = wp_remote_retrieve_body($response);
                            wp_send_json_error("HTTP {$response_code}: {$response_body}");
                        }
                    } else {
                        wp_send_json_error($response->get_error_message());
                    }

                }
                wp_send_json_error("GA endpoint is not defined");
                break;
            case "upload_media":
                $data = $_POST["data"];
                $decoded_data = stripslashes($data);
                $body  = json_decode($decoded_data, true);
                if (empty($body['base64_data']) || empty($body['filename']) || empty($body['mime_type'])) {
                    wp_send_json_error("Empty base64_data or filename or mime_type");
                }

                require_once get_template_directory() . '/includes/content-managers/post-manager.php';
                $post_manager = \WVC_Post_Manager::get_instance();

                $base64_data  = $body['base64_data'];
                $filename     = $body['filename'];
                $content_type = $body['mime_type'];
                // Decode the base64 data
                $file_data = base64_decode($base64_data);
                // Make sure we have a valid filename
                $result = $post_manager->upload_media(
                    $file_data,
                    $filename,
                    $content_type
                );
                wp_send_json_success($result);
                break;
            case "remove_media":
                $data = $_POST["data"];
                $decoded_data = stripslashes($data);
                $body  = json_decode($decoded_data, true);
                if (empty($body['id'])) {
                    wp_send_json_error("Empty id");
                }

                require_once get_template_directory() . '/includes/content-managers/post-manager.php';
                $post_manager = \WVC_Post_Manager::get_instance();

                $result = $post_manager->delete_media($body['id']);
                wp_send_json_success($result);
                break;
            default:
                wp_send_json_error("Invalid action");
        }
    }

    public static function get_menus_to_localize($post_id, $is_editor = false)
    {
        // Build a localized structure mirroring REST API responses
        if ( ! class_exists('WVC_Menu_Manager')) {
            return array(
                'menus'         => array(),
                'menusDetailed' => array(),
            );
        }
        $menu_manager = \WVC_Menu_Manager::get_instance();
        // Matches REST: GET /menus
        $menus = $menu_manager->get_menus();
        // Matches REST: GET /menus/{id} for each menu
        $menus_detailed = array();
        if (is_array($menus)) {
            foreach ($menus as $menu_summary) {
                if (is_array($menu_summary) && isset($menu_summary['id'])) {
                    $menu_full = $menu_manager->get_menu($menu_summary['id']);
                    if ( ! is_wp_error($menu_full)) {
                        $menus_detailed[] = $menu_full;
                    }
                }
            }
        }

        return array(
            'menus'         => $menus,
            'menusDetailed' => $menus_detailed,
        );
    }
}

// Initialize the core functionality
new Core();
